// A driver that simply loads an instance into the Instance class.

// Chris Thiel <cthiel@cse.unl.edu>
// 2008-02-03
// CSCE 821


#include "config.h"
#include "Solver/Instance.h"
#include "Solver/LoadInstance.h"
#include "Solver/Solver.h"
#include "XMLParser/CSPParserCallback.h"
#include "XMLParser/XMLParser.h"
#include <cstdlib>
#include <iostream>
#include <queue>
#include <string>
#include <getopt.h>


static const char *program_name = "csp-solve";


static void
print_usage (std::ostream &out)
{
  out << "Usage: " << program_name << " [OPTION]... FILE\n\n"

"Defaults for the options are specified in brackets.\n\n"

"Options:\n"
"  -S, --solver=ALGORITHM  use solver ALGORITHM [FC-CBJ]:\n"
"                            BT      (chronological backtracking),\n"
"                            FC      (forward checking),\n"
"                            CBJ     (conflict-directed backjumping),\n"
"                            CBJ-cbf (CBJ using chrono. backtracking flag),\n"
"                            FC-CBJ  (FC and CBJ hybrid)\n"
"  -a, --all               find all solutions\n"
"  -H, --heur=HEURISTIC    use variable ordering HEURISTIC [NONE]:\n"
"                            NONE    (no variable ordering)\n"
"                            LD      (least domain),\n"
"                            DEG     (degree),\n"
"                            DDR     (domain-degree ratio)\n"
"  -d, --dvars             use dynamic variable ordering\n"
"                          any consistency algorithms\n"
"  -C, --consist=ALGORITHM use consistency ALGORITHM [NC]:\n"
"                            NC      (node consistency),\n"
"                            PC-2    (path consistency 2)\n"
"                            DPC     (directional path consistency)\n"
"                            PPC     (partial path consistency)\n"
"                            TPPC    (triangle partial path consistency)\n"
"                            TPPC-2  (triangle partial path consistency 2)\n"
"                            If multiple algorithms given, then they are each\n"
"                            performed in the order specified.\n"
"  --no-solve              Stop after running consistency algorithms\n"
"  -v, --verbose           verbose mode\n"
"  -h, --help              display this help and exit\n"
"  -V, --version           display the version and exit\n";
}


static void
print_version (std::ostream &out)
{
  out << PACKAGE_STRING << std::endl;
}


static void
error (const std::string &msg)
{
  std::cerr << program_name << ": " << msg << std::endl;
}


int
main(int argc, char **argv)
{
  struct option options[] =
    {
      { "solver",      1, 0, 'S' },
      { "all",         0, 0, 'a' },
      { "heur",        1, 0, 'H' },
      { "dvar",        0, 0, 'd' },
      { "consist",     1, 0, 'C' },
      { "no-solve",    0, 0, '1' },
      { "verbose",     0, 0, 'v' },
      { "help",        0, 0, 'h' },
      { "version",     0, 0, 'V' },
      { 0, 0, 0, 0 }
    };

  std::string solver_kind;
  bool all_solutions (false);
  std::string var_order_kind;
  bool dynamic_vars (false);
  bool dynamic_vals (false);
  bool no_solve (false);
  bool verbose (false);
  std::queue<CSPSolver::Instance::ConsistencyAlgo> consist_algos;

  // Perform node consistency unconditionally.
  consist_algos.push (CSPSolver::Instance::NC);

  int opt;
  while ((opt = getopt_long (argc, argv, "S:aH:dC:vhV", options, NULL)) != -1)
    {
      switch (opt)
	{
	case 'S':
	  if (!solver_kind.empty ())
	    {
	      error ("Only one solver kind can be specified");
	      print_usage (std::cerr);
	      std::exit (EXIT_FAILURE);
	    }
	  solver_kind = optarg;
	  break;
	case 'a':
	  all_solutions = true;
	  break;
	case 'H':
	  if (!var_order_kind.empty ())
	    {
	      error ("Only one variable ordering heuristic can be specified");
	      print_usage (std::cerr);
	      std::exit (EXIT_FAILURE);
	    }
	  var_order_kind = optarg;
	  break;
	case 'd':
	  dynamic_vars = true;
	  break;
	case 'C':
	  if (std::string (optarg) == "NC")
	    consist_algos.push (CSPSolver::Instance::NC);
	  else if (std::string (optarg) == "PC-2")
	    consist_algos.push (CSPSolver::Instance::PC2);
	  else if (std::string (optarg) == "DPC")
	    consist_algos.push (CSPSolver::Instance::DPC);
	  else if (std::string (optarg) == "PPC")
	    consist_algos.push (CSPSolver::Instance::PPC);
	  else if (std::string (optarg) == "TPPC")
	    consist_algos.push (CSPSolver::Instance::TPPC);
	  else if (std::string (optarg) == "TPPC-2")
	    consist_algos.push (CSPSolver::Instance::TPPC2);
	  else
	    {
	      error (std::string ("Unrecognized consistency algorithm: `")
		     + optarg + '\'');
	      print_usage (std::cerr);
	      std::exit (EXIT_FAILURE);
	    }
	  break;
	case '1':
	  no_solve = true;
	  break;
	case 'h':
	  print_usage (std::cout);
	  std::exit (EXIT_SUCCESS);
	case 'v':
	  verbose = true;
	  break;
	case 'V':
	  print_version (std::cout);
	  std::exit (EXIT_SUCCESS);
	default:
	  print_usage (std::cerr);
	  std::exit (EXIT_FAILURE);
	}
    }

  // Set the variable ordering heuristic.
  CSPSolver::SystematicSearchSolver::VariableOrdering var_order;
  if (var_order_kind.empty () || var_order_kind == "NONE")
    var_order = CSPSolver::SystematicSearchSolver::NO_VAR_ORDER;
  else if (var_order_kind == "LD")
    var_order = CSPSolver::SystematicSearchSolver::LEAST_DOMAIN;
  else if (var_order_kind == "DEG")
    var_order = CSPSolver::SystematicSearchSolver::DEGREE;
  else if (var_order_kind == "DDR")
    var_order = CSPSolver::SystematicSearchSolver::DOMAIN_DEGREE_RATIO;
  else
    {
      error (std::string ("Unrecognized variable ordering heuristic: `")
	     + var_order_kind + '\'');
      print_usage (std::cerr);
      std::exit (EXIT_FAILURE);
    }

  // Set the value ordering heuristic (unused for now).
  CSPSolver::SystematicSearchSolver::ValueOrdering val_order;
  val_order = CSPSolver::SystematicSearchSolver::UNUSED;
  

  // Check that a filename is present.
  if (optind != argc - 1)
    {
      if (optind > argc - 1)
	error ("No file given");
      else
	error ("Extra arguments given");
      print_usage (std::cerr);
      std::exit (EXIT_FAILURE);
    }

  CSPSolver::LoadInstance load;

  try
    {
      // Set up the parser.
      CSPXMLParser::XMLParser<CSPXMLParser::CSPParserCallback> parser(&load);
      parser.setPreferredExpressionRepresentation(CSPXMLParser::AST::TREE);
      
      // Parse the input file.
      parser.parse (argv[optind]);

      // Get the parsed instance.
      CSPSolver::Instance *instance (load.get_instance ());

      // Do consistency checks/algorithms.
      bool keep_going (instance->perform_consistency (consist_algos, verbose));

      // Stop here if we're not interested in solving.
      if (no_solve)
	return 0;

      // Exit early if the problem was found inconsistent.
      if (!keep_going)
	{
	  std::cout << "Inconsistent problem--not solving." << std::endl;
	  return 0;
	}

      // Instantiate the solver.
      CSPSolver::SystematicSearchSolver *solver;
      if (solver_kind == "BT")
	solver = new CSPSolver::BtSolver (instance, var_order, dynamic_vars,
					  val_order, dynamic_vals,
					  all_solutions);
      else if (solver_kind == "FC")
	solver = new CSPSolver::FcSolver (instance, var_order, dynamic_vars,
					  val_order, dynamic_vals,
					  all_solutions);
      else if (solver_kind == "CBJ")
	solver = new CSPSolver::CbjSolver (instance, var_order, dynamic_vars,
					   val_order, dynamic_vals,
					   all_solutions);
      else if (solver_kind == "CBJ-cbf")
	solver = new CSPSolver::CbjCbfSolver (instance, var_order, dynamic_vars,
					      val_order, dynamic_vals,
					      all_solutions);
      else if (solver_kind.empty () || solver_kind == "FC-CBJ")
	solver = new CSPSolver::FcCbjSolver (instance, var_order, dynamic_vars,
					     val_order, dynamic_vals,
					     all_solutions);
      else
	{
	  error (std::string ("Unrecognized solver algorithm: `")
		 + solver_kind + '\'');
	  print_usage (std::cerr);
	  std::exit (EXIT_FAILURE);
	}

      solver->solve ();
      delete solver;
    }
  catch (std::exception &e)
    {
      std::cout.flush();
      std::cerr << "\n\tUnexpected exception: " << e.what() << std::endl;
      std::exit (EXIT_FAILURE);
    }
  
  return 0;
}
