// -*- C++ -*-

// Load an instance into an Instance class.

// Chris Thiel <cthiel@cse.unl.edu>
// 2008-02-03
// CSCE 821


#ifndef _LoadInstance_h_
#define _LoadInstance_h_

#include "Constraint.h"
#include "Instance.h"
#include "Value.h"
#include "XMLParser/CSPParserCallback.h"
#include "XMLParser/ExpressionParser.h"
#include <cassert>
#include <set>
#include <vector>


namespace CSPSolver
{
  // Definition of the functions which are called by the parser when
  // it reads new information in the file.  These functions are in
  // charge of creating the data structure that the solver needs to do
  // its job.
  class LoadInstance : public CSPXMLParser::CSPParserCallback
  {
    typedef std::vector<Domain *> DomainVector;
    typedef std::vector<Variable *> VariableVector;
    typedef std::vector<Relation *> RelationVector;
    typedef std::vector<Predicate *> PredicateVector;
    typedef std::vector<Constraint *> ConstraintVector;

  private:
    // This is the Instance data structure loaded by this class.
    Instance *instance;

    // The name of this instance.
    std::string instance_name;

    // A vector of domains.  These are not used directly outside of
    // this class.
    DomainVector domain_vector;

    // This is the id of the domain being processed.
    DomainVector::size_type current_domain_id;

#ifndef NDEBUG
    // Members used only in assertion checking: The size of the
    // current domain and the number of values added to the domain so
    // far.
    int current_domain_size;
    int current_domain_value_count;
#endif

    // A vector of variables.
    VariableVector variable_vector;

    // A vector of relations.
    RelationVector relation_vector;

    // This is the id of the relation being processed.
    RelationVector::size_type current_relation_id;

#ifndef NDEBUG
    // Members used only in assertion checking: The size of the
    // current relation and the number of tuples added to the relation
    // so far.
    int current_relation_size;
    int current_relation_tuple_count;
#endif

    // A vector of relations.
    PredicateVector predicate_vector;

    // This is the id of the relation being processed.
    PredicateVector::size_type current_predicate_id;

    // A vector of pointers to constraints.
    ConstraintVector constraint_vector;

    // The id of the constraint being processed.
    ConstraintVector::size_type current_constraint_id;

  public:

    LoadInstance (void)
      : instance (NULL),
	instance_name (),
	domain_vector (),
	current_domain_id (),
#ifndef NDEBUG
	current_domain_size (0),
	current_domain_value_count (0),
#endif
	variable_vector (),
	relation_vector (),
	current_relation_id (),
#ifndef NDEBUG
	current_relation_size (0),
	current_relation_tuple_count (0),
#endif
	predicate_vector (),
	current_predicate_id (0),
	constraint_vector (),
	current_constraint_id (0)
    {}

    Instance *
    get_instance (void)
    {
      assert (instance != NULL);
      return instance;
    }

    virtual void
    beginInstance (const std::string &name)
    {
      instance_name = name;
      
      // Instantiate the instance.
//       instance = new Instance (name);
    }

    virtual void
    beginDomainsSection (int nbDomains) 
    {  
      assert (domain_vector.empty ());

      // Resize the domain vector to accommodate all of the domains.
      domain_vector.resize (nbDomains);
    }
  
    virtual void 
#ifndef NDEBUG
    beginDomain (const std::string & /* name */, int idDomain,
		 int nbValues)
#else
    beginDomain (const std::string & /* name */, int idDomain,
		 int /* nbValues */)
#endif
    {
      assert (idDomain >= 0);
      assert ((unsigned int) (idDomain) < domain_vector.size ());
      assert ((unsigned int) (idDomain) >= current_domain_id);

      // Set the current domain id so we can reference this domain
      // below.
      current_domain_id = idDomain;

      // Create a new domain, and store a pointer to it in the domain
      // vector.
      domain_vector[current_domain_id] = new Domain ();
//       domain_vector[current_domain_id] = new Domain (name, idDomain);

#ifndef NDEBUG
      // For assertion checking in lower methods.
      current_domain_size = nbValues;
      current_domain_value_count = 0;
#endif
    }

    void 
    addDomainValue (int v) 
    {
      assert (current_domain_id <= domain_vector.size ());
      assert (domain_vector[current_domain_id] != NULL);
      assert ((unsigned int) (current_domain_size)
	      > domain_vector[current_domain_id]->size ());
      assert (current_domain_value_count < current_domain_size);

      // Insert V into the current domain using the end as a hint.
      Domain *current_domain = domain_vector[current_domain_id];
      current_domain->insert (current_domain->end (), v);

#ifndef NDEBUG
      // For assertion checking in subsequent calls to this method.
      ++current_domain_value_count;
#endif
    }

    virtual void 
    addDomainValue (int first, int last) 
    {
      assert (first <= last);

      // Insert the values in the range [FIRST, LAST] into the current
      // domain.
      for (int v = first; v <= last; ++v)
	addDomainValue (v);
    }

    virtual void 
    endDomain (void) 
    {
      // Unused.
    }

    virtual void 
    endDomainsSection (void) 
    {
      assert (current_domain_id <= domain_vector.size ());
    }

    virtual void 
    beginVariablesSection (int nbVariables) 
    {
      assert (nbVariables >= 0);
      assert ((unsigned int) (nbVariables) < variable_vector.max_size ());
      assert (variable_vector.empty ());

      // Resize the variable vector to accommodate all of the
      // variables.
      variable_vector.resize (nbVariables);
    }
  
    virtual void
    addVariable (const std::string &name, int idVar,
		 const std::string &/* domain */, int idDomain)
    {
      assert (!name.empty ());
      assert (idVar >= 0);
      assert (idDomain >= 0);
      assert ((unsigned int) (idDomain) <= current_domain_id);

      // Create a new variable.
      Variable *var = new Variable (name, idVar, *domain_vector[idDomain]);

      // Store a pointer to the new variable in the relation vector.
      variable_vector[idVar] = var;
    }

    virtual void
    endVariablesSection (void) 
    {
//       instance->set_variables (variable_vector);
    }

    virtual void
    beginRelationsSection (int nbRelations) 
    {
      assert (relation_vector.empty ());
      
      // Resize the relation vector to accommodate all of the
      // relations.
      relation_vector.resize (nbRelations);
    }
  
    virtual void
#ifndef NDEBUG
    beginRelation (const std::string &name, int idRel, int arity,
		   int nbTuples, bool isSupport)
#else
    beginRelation (const std::string &name, int idRel, int arity,
		   int /* nbTuples */, bool isSupport)
#endif
    {
      assert (idRel >= 0);
      assert ((unsigned int) (idRel) < relation_vector.size ());
      assert ((unsigned int) (idRel) >= current_relation_id);
      assert (arity > 0);
      assert (nbTuples >= 0);

      // Set the current relaton id so we can reference this relation
      // below.
      current_relation_id = idRel;

      // Create a new relation, and store a pointer to it in the
      // relation vector.
      relation_vector[current_relation_id] = new Relation (arity, isSupport,
							   name);

#ifndef NDEBUG
      // For assertion checking in lower methods.
      current_relation_size = nbTuples;
      current_relation_tuple_count = 0;
#endif
    }

    virtual void 
    addRelationTuple (int arity, int tuple[]) 
    {
      assert (arity > 0);
      assert (arity_t (arity)
	      == relation_vector[current_relation_id]->get_arity ());
      assert (current_relation_id <= relation_vector.size ());
      assert (RelationVector::size_type (current_relation_size)
	      > relation_vector[current_relation_id]->size ());
      assert (current_relation_tuple_count < current_relation_size);

      // Create a tuple for the current relation.
      ValueVector value_tuple;
      for (int i=0; i < arity; ++i)
	value_tuple.push_back (tuple[i]);
      
      // Insert the tuple into the current relation using the end as a
      // hint.
      Relation *current_relation = relation_vector[current_relation_id];
      current_relation->insert (current_relation->end (), value_tuple);

#ifndef NDEBUG
      ++current_relation_tuple_count;
#endif
    }

    virtual void
    endRelation (void) 
    {
      // Unused.
    }

    virtual void
    endRelationsSection (void) 
    {
      assert (current_relation_id <= relation_vector.size ());
    }

    virtual void
    beginPredicatesSection (int nbPredicates) 
    {
      assert (predicate_vector.empty ());

      // Resize the predicate vector to accommodate all of the
      // predicates.
      predicate_vector.resize (nbPredicates);
    }
  
    virtual void
    beginPredicate (const std::string &name, int idPred) 
    {
      assert (idPred >= 0);

      // Set the current predicate id so we can reference this
      // predicate below.
      current_predicate_id = idPred;

      // Create a new predicate, and store a pointer to it in the
      // predicate vector.
      predicate_vector[current_predicate_id] = new Predicate (name);
    }

    virtual void 
    addFormalParameter(int pos, const std::string & /* name */,
		       const std::string & /* type */) 
    {
      predicate_vector[current_predicate_id]->reserve_formal_param (pos);
    }

    virtual void 
    predicateExpression(CSPXMLParser::AST *tree) 
    {
      predicate_vector[current_predicate_id]->set_expression (tree);
    }

    virtual void predicateExpression(const std::string & /* expr */) 
    {
      // Unused; unreachable.
      assert (false);
    }

    virtual void endPredicate() 
    {
      // Unused.
    }

    virtual void endPredicatesSection() 
    {
      // Unused.
    }

    virtual void 
    beginConstraintsSection (int nbConstraints) 
    {
      assert (constraint_vector.empty ());
      constraint_vector.resize (nbConstraints);
    }
  
    virtual void 
    beginConstraint (const std::string &name, int idConstr, int arity, 
		     const std::string & /*reference */, 
		     CSPXMLParser::CSPDefinitionType type, int id)
    {
      assert (idConstr >= 0);
      assert ((unsigned int) (idConstr) < constraint_vector.size ());
      assert ((unsigned int) (idConstr) >= current_constraint_id);
      assert (arity > 0);
      assert (type <= CSPXMLParser::UndefinedType);
      assert (id >= 0);

      // Create a constraint of the appropriate kind.
      Constraint *constraint;
      switch (type)
	{
	case CSPXMLParser::RelationType:
	  {
	    Relation *relation = relation_vector[id];
	    if (relation->support_p ())
	      {
		constraint = new SupportConstraint (arity, relation, name);
	      }
	    else
	      {
		constraint = new ConflictConstraint (arity, relation, name);
	      }
	  }
	  break;
	case CSPXMLParser::PredicateType:
	  constraint = new IntensionConstraint (arity, predicate_vector[id],
						name);
	  break;
	case CSPXMLParser::GlobalConstraintType:
	case CSPXMLParser::DomainType:
	case CSPXMLParser::VariableType:
	case CSPXMLParser::ConstraintType:
	case CSPXMLParser::UndefinedType:
	  // These are unused for now; fall through.
	default:
	  // Unreachable.
	  assert (false);
	}

      // Set the current contraint id and store a pointer to the newly
      // created constraint in the constaint vector.
      current_constraint_id = idConstr;
      constraint_vector[current_constraint_id] = constraint;
    }

    virtual void 
    addVariableToConstraint (const std::string & /* name */, int idVar) 
    {
      assert (idVar >= 0);
      assert ((unsigned int) (idVar) < variable_vector.size ());

      // Lookup the variable based on its id.
      Variable *var = variable_vector[idVar];
      assert (var != NULL);

      // Lookup the current constraint.
      Constraint *current_constraint 
	= constraint_vector[current_constraint_id];
      assert (current_constraint != NULL);

      // Add VAR to the current constraint.
      current_constraint->add_variable (var);
    }

    virtual void
    addEffectiveParameter (int /* pos */, const std::string & /* expr */) 
    {
      // Unused; unreachable.
      assert (false);
    }

    virtual void 
    addEffectiveParameter (int /* pos */, CSPXMLParser::AST * /* tree */) 
    {
      // Unused; unreachable.
      assert (false);
    }

    virtual void 
    addEffectiveParameterVariable (int pos, const std::string & /* name */,
				   int idVar)
    {
      IntensionConstraint *current_constraint 
	= dynamic_cast<IntensionConstraint *> 
	(constraint_vector[current_constraint_id]);
      current_constraint->set_param_variable (pos, variable_vector[idVar]);
    }

    virtual void 
    addEffectiveParameterInteger (int pos, int value)
    {
      IntensionConstraint *current_constraint 
	= dynamic_cast<IntensionConstraint *> 
	(constraint_vector[current_constraint_id]);
      current_constraint->set_param_value (pos, value);
    }

    virtual void 
    addEffectiveParameterList (int /* pos */) 
    {
      // Unused, unreachable.
      assert (false);
    }

    virtual void
    beginList (void) 
    {
      // Unused, unreachable.
      assert (false);
    }

    virtual void 
    addListVariable (int /* pos */, const std::string & /* name */,
		     int /* idVar */) 
    {
      // Unused, unreachable.
      assert (false);
    }

    virtual void 
    addListInteger (int /* pos */, int /* value */) 
    {
      // Unused, unreachable.
      assert (false);
    }

    virtual void endList (void) 
    {
      // Unused, unreachable.
      assert (false);
    }

    virtual void
    endConstraint (void) 
    {
      // Unused.
    }

    /**
     * end the definition of all constraints
     */
    virtual void
    endConstraintsSection (void) 
    {
//       assert (instance != NULL);
//       instance->set_constraints (constraint_vector);
    }

    virtual void 
    endInstance (void) 
    {
      assert (instance == NULL);
      instance = new Instance (variable_vector.begin (), variable_vector.end (),
			       constraint_vector.begin (),
			       constraint_vector.end (), instance_name);
    }
  };
} // namespace CSPSolver

#endif	// _LoadInstance_h_
