// -*- C++ -*-

// CSP instance class.

// Chris Thiel <cthiel@cse.unl.edu>
// 2008-02-02
// CSCE 821


#ifndef _Instance_h_
#define _Instance_h_

#include "Constraint.h"
#include "Value.h"
#include "Variable.h"
#include <algorithm>
#include <cassert>
#include <iterator>
#include <map>
#include <queue>
#include <string>
#include <utility>
#include <vector>


namespace CSPSolver
{
  struct ConsistencyStats
  {
    ConsistencyStats (void);
    clock_t begin, end;
  };

  enum NcP { NC_P_YES, NC_P_NO, NC_P_MAYBE };

  struct NcStats : public ConsistencyStats
  {
    NcStats (void);
    NcP nc_p;
  };

  enum PcP { PC_P_YES, PC_P_NO, PC_P_MAYBE };

  struct Edge : public std::pair<Variable *, Variable *>
  {
    Edge (Variable * first, Variable *second)
      : std::pair<Variable *, Variable *> (first, second)
    {}
  };

  std::ostream &operator<< (std::ostream &out, const Edge &edge);

  struct PcStats : public ConsistencyStats
  {
    PcStats (void);
    void add_elim (const Edge &edge) 
    {
      ++elim_count[edge];
    }

    unsigned int rev3;
    unsigned int rev;
    unsigned int cc;
    PcP pc_p;
    std::map<Edge, unsigned int> elim_count;
  };

  std::ostream &operator<< (std::ostream &out, const ConsistencyStats &stats);
  std::ostream &operator<< (std::ostream &out, const NcStats &stats);
  std::ostream &operator<< (std::ostream &out, const PcStats &stats);

  typedef VariableSet Clique;
  typedef std::vector<Clique> CliqueVector;

  std::ostream &operator<< (std::ostream &out, const Clique &clique);
  
  // An edge in the joint tree.
  class JoinTreeEdge : public std::pair<CliqueVector::const_iterator,
					CliqueVector::const_iterator>
  {
  public:
    JoinTreeEdge (CliqueVector::const_iterator m,
		  CliqueVector::const_iterator n);
  };
  
  std::ostream &operator<< (std::ostream &out, const JoinTreeEdge &edge);

  typedef std::vector<JoinTreeEdge> JoinTreeEdgeVector;

  // A jooin-tree.  The first clique in NODES is considered the root.
  class JoinTree
  {
  public:
    typedef std::map<CliqueVector::const_iterator,
		     CliqueVector::const_iterator> ParentMap;
    typedef std::map<CliqueVector::const_iterator,
		     std::vector<CliqueVector::const_iterator> > ChildrenMap;
    typedef std::vector<CliqueVector::const_iterator> VisitOrder;

    JoinTree (const CliqueVector &cliques,
	      const std::vector<std::pair<Clique, Clique> > &clique_pairs);

    CliqueVector::size_type
    node_count (void) const;

    std::vector<JoinTreeEdge>::size_type
    edge_count (void) const;
    
    CliqueVector::const_iterator
    get_root (void) const;

    const std::vector<CliqueVector::const_iterator> &
    get_leaves (void) const;

    CliqueVector::const_iterator
    get_parent (CliqueVector::const_iterator node) const;

    const std::vector<CliqueVector::const_iterator> &
    get_children (CliqueVector::const_iterator node) const;

    const VisitOrder &
    get_postorder (void) const;

  private:
    CliqueVector nodes;
    JoinTreeEdgeVector edges;
    ParentMap parent_map;
    ChildrenMap children_map;
    std::vector<CliqueVector::const_iterator> leaves;
    VisitOrder postorder;
  };

  // Forward declarations.
  class Constraint;
  class SetTriangle;
  class SetEdge;
			   
  // Map: {i, j, k} -> bool.
  typedef std::map<SetTriangle, bool> EnqueuedTriagnleMap;

  // Vector: [{i, j, k}, ..., {x, y, z}]
  typedef std::vector<SetTriangle> SetTriangleVector;

  // Map: {m, n} -> [{k, m, n}, ..., {m, n, z}].
  typedef std::map<SetEdge, SetTriangleVector> RelatedTrianglesMap;


  class Instance
  {
  public:
    template <class VariableInputIterator, class ConstraintInputIterator>
    Instance (VariableInputIterator first_var, VariableInputIterator last_var,
	      ConstraintInputIterator first_constr,
	      ConstraintInputIterator last_constr,
	      const std::string &name = "");
    ~Instance (void);
    const VariableVector &get_variables (void) const;
    const std::vector<const Constraint *> &get_constraints (void) const;
    bool is_edge (const Variable *m, const Variable *n) const;
    bool is_empty_edge (const Variable *m, const Variable *n) const;
    enum ConsistencyAlgo { NC, PC2, DPC, PPC, TPPC, TPPC2 };
    VariableVector perform_min_fill (bool verbose = false);
    void complete_paths_of_length_two (bool verbose = false);
    CliqueVector cliques (const VariableVector &peo, bool verbose = false);
    JoinTree join_tree (const CliqueVector &cliques, bool verbose = false);
    bool perform_consistency (ConsistencyAlgo algo, ConsistencyStats *stats,
			      bool verbose = false);
    bool perform_consistency (std::queue<ConsistencyAlgo> algos,
			      bool verbose = false);
    bool check (const VvpSet &vvps, unsigned int *cc = NULL) const;

  private:
    VariableVector variables;
    std::vector<const Constraint *> constraints;
    std::map<Scope, Constraint *> constraint_map;
    const std::string name;

    void add_constraint (Constraint *constr);
    bool revise (Variable *x, Variable *y);
    bool revise_3 (Variable *x, Variable *y, Variable *z, PcStats *stats);
    bool perform_node_consistency (NcStats *stats);
    bool perform_path_consistency_2 (PcStats *stats);
    bool perform_directional_path_consistency (PcStats *stats);
    bool perform_partial_path_consistency (PcStats *stats);
    bool tppc (std::queue<SetTriangle> &tri_queue, EnqueuedTriagnleMap &enq_map,
	       const RelatedTrianglesMap &related_triangles, PcStats *stats);
    bool perform_triangle_partial_path_consistency (PcStats *stats);
    bool perform_triangle_partial_path_consistency_2 (const JoinTree &tree,
						      PcStats *stats);

    
    // Friends:
    friend std::ostream &operator<< (std::ostream &out,
				     const Instance &instance);
  };


  // Inline definitions.


  inline
  ConsistencyStats::ConsistencyStats (void)
    : begin (0),
      end (0)
  {}


  inline
  NcStats::NcStats (void)
    : nc_p (NC_P_MAYBE)
  {}

  
  inline
  PcStats::PcStats (void)
    : rev3 (0),
      rev (0),
      cc (0),
      pc_p (PC_P_MAYBE)
  {}


  inline
  JoinTreeEdge::JoinTreeEdge (CliqueVector::const_iterator m,
			      CliqueVector::const_iterator n)
    : std::pair<CliqueVector::const_iterator,
		CliqueVector::const_iterator> (m, n)
  {
    assert (m < n);
    assert (*m != *n);
  }


  inline CliqueVector::size_type
  JoinTree::node_count (void) const
  {
    return nodes.size ();
  }


  inline std::vector<JoinTreeEdge>::size_type
  JoinTree::edge_count (void) const
  {
    return edges.size ();
  }


  inline CliqueVector::const_iterator
  JoinTree::get_root (void) const
  {
    assert (!nodes.empty ());
    return nodes.begin ();
  }


  inline const std::vector<CliqueVector::const_iterator> &
  JoinTree::get_leaves (void) const
  {
    return leaves;
  }


  inline CliqueVector::const_iterator
  JoinTree::get_parent (CliqueVector::const_iterator node) const
  {
    assert (parent_map.find (node) != parent_map.end ());
    return parent_map.find (node)->second;
  }


  inline const std::vector<CliqueVector::const_iterator> &
  JoinTree::get_children (CliqueVector::const_iterator node) const
  {
    assert (children_map.find (node) != children_map.end ());
    return children_map.find (node)->second;
  }


  inline const JoinTree::VisitOrder &
  JoinTree::get_postorder (void) const
  {
    return postorder;
  }


  template <class VariableInputIterator, class ConstraintInputIterator>
  inline
  Instance::Instance (VariableInputIterator first_var,
		      VariableInputIterator last_var,
		      ConstraintInputIterator first_constr,
		      ConstraintInputIterator last_constr,
		      const std::string &name_)
    : variables (first_var, last_var),
      constraints (),
      constraint_map (),
      name (name_)
  {
    assert (!name.empty ());

    // Add the constraints.
    for (ConstraintInputIterator c = first_constr; c != last_constr; ++c)
      add_constraint (*c);
  }


  inline const VariableVector &
  Instance::get_variables (void) const
  {
    return variables;
  }


  inline const std::vector<const Constraint *> &
  Instance::get_constraints (void) const
  {
    return constraints;
  }
} // namespace CSPSolver

#endif	// _Instance_h_
