// CSP instance methods and functions.

// Chris Thiel <cthiel@cse.unl.edu>
// 2008-02-28
// CSCE 821


#include "config.h"
#include "Instance.h"
#include "Constraint.h"
#include "Value.h"
#include "Variable.h"
#include <algorithm>
#include <cassert>
#include <ctime>
#include <deque>
#include <iostream>
#include <queue>
#include <utility>
#include <vector>


namespace CSPSolver
{
  // Add the constraint CONSTR to this instance.
  void
  Instance::
  add_constraint (Constraint *constr)
  {
    assert (constr != NULL);
    const Scope &scope (constr->get_scope ());

    std::map<Scope, Constraint *>::const_iterator
      mi (constraint_map.find (scope));
    if (mi == constraint_map.end ())
      {
	// No constraint has been added with the same scope as CONSTR,
	// so simply add CONSTR to CONSTRAINTS and CONSTRAINT_MAP.
	constraints.push_back (constr);
	constraint_map.insert (std::make_pair (scope, constr));
      }
    else
      {
	// Another constraint has already been added that shares the
	// same scope as CONSTR, so merge CONSTR into it.  We don't
	// need to add anything to CONSTRAINTS or CONSTRAINT_MAP.
	assert (mi->second != NULL);
	mi->second->merge (constr);
      }
  }


  // Filter domains to make this network node-consistent.  Return true
  // iff the network is node-consistent.  If this method returns
  // false, then the network is inconsistent and cannot be solved.
  // After calling this method, all unary constraints are removed if
  // this instance is node-consistent.  If it is not node-consistent,
  // then some unary constraints may remain.
  bool
  Instance::
  perform_node_consistency (NcStats *stats)
  {
    if (stats)
      stats->begin = std::clock ();

    // For each constraint C in the network:
    for (std::map<Scope, Constraint *>::iterator
	   mi (constraint_map.begin ()); mi != constraint_map.end (); )
      {
	const Constraint *c (mi->second);
	assert (c != NULL);
	
	// Skip the constraint if it is not unary.
	if (!c->unary_p ())
	  {
	    ++mi;
	    continue;
	  }

	// The single variable in the scope of this constraint.
	Variable *v (*c->get_scope ().begin ());
	assert (v != NULL);

	// The domain of the single variable.
	Domain &dom (v->get_initial_domain ());

	// For each value in the domain of the single variable:
	for (Domain::const_iterator di (dom.begin ()); di != dom.end (); )
	  {
	    // See if the domain value is consistent.
	    VvpSet vvp_singleton (v, *di);
	    const bool consistent = c->check (vvp_singleton);

	    // If the domain value is inconsistent with the
	    // constraint, then remove it.
	    if (!consistent)
	      dom.erase (di++);
	    else
	      ++di;
	  }

	// If a domain wipe-out occurred, then the network is not
	// node-consistent.
	if (dom.empty ())
	  {
	    if (stats)
	      {
		stats->nc_p = NC_P_NO;
		stats->end = std::clock ();
	      }
	    return false;
	  }

	// Remove this constraint from this instance.
	assert (std::find (constraints.begin (), constraints.end (), c)
		!= constraints.end ());
	constraints.erase (std::remove (constraints.begin (),
					constraints.end (), c));
	assert (std::find (constraints.begin (), constraints.end (), c)
		== constraints.end ());

	// Remove this constraint from the constraint map of this
	// instance.
	assert (constraint_map.find (Scope (v)) != constraint_map.end ());
	constraint_map.erase (mi++);
	assert (constraint_map.find (Scope (v)) == constraint_map.end ());

	// Delete the constraint, it is no longer needed.  This also
	// updates the set of constraints for each variable in its
	// scope.
	delete c;
      }

    // If we made it here, then the network is node-consistent.
    if (stats)
      {
	stats->nc_p = NC_P_YES;
	stats->end = std::clock ();
      }
    return true;
  }


  // Given a variable X and a second variable Y, modify the domain
  // D(X), such that X is arc-consistent relative to Y.  Return true
  // iff D(X) is modified, that is iff D'(X) is a proper subset of
  // D(X).  Based on Revise in Rina Dechter's Constraint Processing
  // (p. 55).
  bool
  Instance::
  revise (Variable *x, Variable *y)
  {
    assert (x != NULL);
    assert (y != NULL);

    bool domain_modified (false);

    // For each value A in D(X):
    for (Domain::const_iterator ai (x->get_initial_domain ().begin ());
	 ai != x->get_initial_domain ().end (); )
      {
	bool x_supported_by_y (false);

	// For each value B in D(Y):
	for (Domain::iterator bi (y->get_initial_domain ().begin ());
	     bi != y->get_initial_domain ().end (); ++bi)
	  {
	    // See if the VVPs {X=A, Y=B} are consistent.
	    const VvpSet xy (x, *ai, y, *bi);
	    const bool xy_consistent (check (xy, NULL));

	    // If the VVPs are consistent, then Y=B supports X=A.
	    if (xy_consistent)
	      {
		x_supported_by_y = true;
		break;
	      }
	  }

	// If we found no value in D(Y) to support X=A, then remove A
	// from D(X).
	if (!x_supported_by_y)
	  {
	    x->get_initial_domain ().erase (*ai++);
	    domain_modified = true;
	  }
	else
	  {
	    ++ai;
	  }
      }

    return domain_modified;
  }


  // Given a pair of variables (X,Y) and a third variable Z, modify
  // the constraint R(X,Y) such that R'(X,Y) is the loosest constraint
  // that satisfies path consistency.  Return true iff R(X,Y) is
  // modified, that is iff R(X,Y) != R'(X,Y).  Based on Revise-3 in
  // Rina Dechter's Constraint Processing (p. 62).
  bool
  Instance::
  revise_3 (Variable *x, Variable *y, Variable *z, PcStats *stats)
  {
    assert (x != NULL);
    assert (y != NULL);
    assert (z != NULL);

    bool constraint_modified (false);
    Relation *conf_relation (NULL);

    if (stats)
      ++(stats->rev3);

    // std::cerr << "Revise-3 (" << Edge (x, y) << "," << *z << ')' << std::endl;

    // For each pair (A,B) in R(X,Y):
    for (Domain::const_iterator a (x->get_initial_domain ().begin ());
	 a != x->get_initial_domain ().end (); ++a)
      for (Domain::const_iterator b (y->get_initial_domain ().begin ());
	   b != y->get_initial_domain ().end (); ++b)
	{
	  unsigned int cc;
	  
	  // See if the VVP {X=A, Y=B} is consistent.
	  const VvpSet xy (x, *a, y, *b);
	  const bool xy_consistent (check (xy, stats ? &cc : NULL));
	  if (stats)
	    stats->cc += cc;

	  // If the VVP is consistent, then (A,B) in R(X,Y).
	  if (xy_consistent)
	    {
	      bool xy_supported_by_z (false);

	      // For each value C in D(Z):
	      for (Domain::iterator c (z->get_initial_domain ().begin ());
		   c != z->get_initial_domain ().end (); ++c)
		{
		  // See if the VVP sets {X=A, Z=C} and {Y=B, Z=C} are
		  // consistent.
		  const VvpSet xz (x, *a, z, *c);
		  const VvpSet yz (y, *b, z, *c);
		  const bool xz_consistent (check (xz, stats ? &cc : NULL));
		  if (stats)
		    stats->cc += cc;
		  if (!xz_consistent)
		    continue;

		  const bool yz_consistent (check (yz, stats ? &cc : NULL));
		  if (stats)
		    stats->cc += cc;
		  if (!yz_consistent)
		    continue;

		  // If both VVP sets are consistent, then Z=C
		  // supports {X=A, Y=B} and we can quit looking.
		  assert (xz_consistent && yz_consistent);
		  xy_supported_by_z = true;
		  break;
		}

	      // If we found no value in D(Z) to support {X=A, Y=B},
	      // then remove {X=A, Y=B} from R(X,Y).
	      if (!xy_supported_by_z)
		{
		  // Create a new conflict relation if one does not
		  // already exist.
		  if (!conf_relation)
		    conf_relation = new Relation (2, false, "confl_relation");
		  
		  // Insert the tuple (A,B) into the conflict
		  // relation.  All unsupported tuples will be removed
		  // at the same time later.
		  ValueVector conflicts;
		  // Note: these must be pushed in the order A, B.
		  conflicts.push_back (*a);
		  conflicts.push_back (*b);
		  conf_relation->insert (conflicts);
		  constraint_modified = true;

		  if (stats)
		    {
		      ++(stats->rev);
		      assert (x->get_name () != y->get_name ());
		      Variable *min = ((x->get_name () < y->get_name ())
				       ? x :  y);
		      Variable *max = ((x->get_name () < y->get_name ())
				       ? y :  x);
		      assert (min != max);
			
		      stats->add_elim (Edge (min, max));

		      // std::cerr << "Eliminating (" << *a << "," << *b 
		      // 		<< ") from " << Edge (min, max)
		      // 		<< " by " << *z << std::endl;
		    }
		}
	    }
	}
    
    // If we created a new relation, then we need to make a constraint
    // and add it to this problem and the variables in its scope.
    if (conf_relation)
      {
	assert (constraint_modified);

	// Create a new conflict constraint using the conflict
	// relation.  This constraint represents the VVPs {X=A, Y=B}
	// for all unsupported tuples (A, B).
	ConflictConstraint *conf_constr
	  = new ConflictConstraint (2, conf_relation, "confl_constraint");
	// Note: these must be pushed in the order X, Y.
	conf_constr->add_variable (x);
	conf_constr->add_variable (y);

	// Create the scope {X, Y} for looking up all constraints with
	// such a scope.
	Scope xy (x, y);
	
	// We handle things differently depending whether a constraint
	// over {X, Y} already exists.
	std::map<Scope, Constraint *>::const_iterator
	  mi (constraint_map.find (xy));
	if (mi == constraint_map.end ())
	  {
	    // There are no explicit constraints with scope {X, Y}.
	    // The conflict constraint replaces the universal
	    // constraint on {X, Y}.

	    // Handle just like a adding regular constraint.
	    constraints.push_back (conf_constr);
	    constraint_map.insert (std::make_pair (xy, conf_constr));
	  }
	else
	  {
	    // There already exists a constraint with scope {X, Y}, so
	    // merge the conflict constraint into it.
	    assert (mi->first == xy);
	    assert (mi->second != NULL);
	    mi->second->merge (*conf_constr);
	    
	    // Note: we can only delete the conflict constraint and
	    // conflict relation when merging because it is not needed
	    // after a merge.
	    delete conf_constr;
	    delete conf_relation;
	  }
      }

    return constraint_modified;
  }


  // A triangle represented as a set of Variable pointers.
  class SetTriangle : public std::set<Variable *>
  {
  public:
    // Constructor.  I, J, and K must be non-NULL and unique.
    SetTriangle (Variable *i, Variable *j, Variable *k)
    {
      assert (i != NULL);
      assert (j != NULL);
      assert (k != NULL);
      assert (i != j);
      assert (i != k);
      assert (j != k);

      insert (i);
      insert (j);
      insert (k);
    }

    // Set the values of I, J, and K by reference.
    void
    get_variables (Variable *&i, Variable *&j, Variable *&k) const
    {
      assert (size () == 3);

      const_iterator ti (begin ());
      i = *ti++;
      j = *ti++;
      k = *ti++;
      assert (ti == end ());
    }
  };


  std::ostream &
  operator<< (std::ostream &out, const SetTriangle &triangle)
  {
    assert (triangle.size () == 3);
    out << '{';
    for (SetTriangle::const_iterator ti (triangle.begin ());
	 ti != triangle.end (); ++ti)
      {
	if (ti != triangle.begin ())
	  out << ", ";
	out << **ti;
      }
    out << '}';
    return out;
  }


  std::ostream &
  operator<< (std::ostream &out, const EnqueuedTriagnleMap &map)
  {
    for (EnqueuedTriagnleMap::const_iterator mi (map.begin ());
	 mi != map.end (); ++mi)
      out << mi->first << " -> " << mi->second << std::endl;
    return out;
  }

  // Just like SetTriangle, except we are representing edges instead
  // of triangles.
  class SetEdge : public std::set<Variable *>
  {
  public:
    SetEdge (Variable *m, Variable *n)
    {
      assert (m != NULL);
      assert (n != NULL);
      assert (m != n);

      insert (m);
      insert (n);
    }

    void
    get_variables (Variable *&m, Variable *&n) const
    {
      assert (size () == 2);

      const_iterator ei (begin ());
      m = *ei++;
      n = *ei++;
      assert (ei == end ());
    }
  };


  std::ostream &
  operator<< (std::ostream &out, const SetEdge &edge)
  {
    assert (edge.size () == 2);
    out << '{';
    for (SetEdge::const_iterator ei (edge.begin ());
	 ei != edge.end (); ++ei)
      {
	if (ei != edge.begin ())
	  out << ", ";
	out << **ei;
      }
    out << '}';
    return out;
  }


  // A variable triplet (a triangle).  We use iterators instead of
  // pointers since it makes it easy to do relational comparisons
  // between the variables as they are ordered in the search path.
  template<typename Type>
  struct Triple
  {
    Triple (Type _first, Type _second, Type _third)
      : first (_first), second (_second), third (_third)
    {}
    Type first;
    Type second;
    Type third;
  };

  typedef Triple<Variable *> VariableTriple;
  typedef Triple<VariableVector::iterator> IteratorTriple;


  template<typename Type>
  std::ostream &
  operator<< (std::ostream &out, const Triple<Type> &triple)
  {
    return out << '(' << *triple.first << ',' << *triple.second << ','
  	       << *triple.third << ')';
  }


  // bool
  // operator== (const IteratorTriple &left, const IteratorTriple &right)
  // {
  //   return (left.first == right.first && left.second == right.second
  // 	    && left.third == right.third);
  // }


  static bool
  pc_false (PcStats *stats)
  {
    if (stats)
      {
	stats->pc_p = PC_P_NO;
	stats->end = std::clock ();
      }
    return false;
  }


  static bool
  pc_true (PcStats *stats)
  {
    if (stats)
      {
	stats->pc_p = PC_P_YES;
	stats->end = std::clock ();
      }
    return true;
  }


  static bool
  pc_maybe (PcStats *stats)
  {
    if (stats)
      {
	// We don't update pc_p in case it was already determined.
	stats->end = std::clock ();
      }
    return true;
  }
  

  typedef std::map<CliqueVector::const_iterator, bool> JoinTreeVisitedNodeMap;


  static void
  find_parents_and_children_1 (const JoinTreeEdgeVector &edges,
			       JoinTree::ParentMap &parent_map,
			       JoinTree::ChildrenMap &children_map,
			       JoinTreeVisitedNodeMap &visited,
			       JoinTree::VisitOrder &postorder,
			       CliqueVector::const_iterator node)
  {
    visited[node] = true;
    for (JoinTreeEdgeVector::const_iterator ei (edges.begin ());
	 ei != edges.end (); ++ei)
      {
	if (ei->first == node && !visited[ei->second])
	  {
	    children_map[node].push_back (ei->second);
	    parent_map[ei->second] = node;
	    find_parents_and_children_1 (edges, parent_map, children_map,
					 visited, postorder, ei->second);
	  }
	else if (ei->second == node && !visited[ei->first])
	  {
	    children_map[node].push_back (ei->first);
	    parent_map[ei->first] = node;
	    find_parents_and_children_1 (edges, parent_map, children_map,
					 visited, postorder, ei->first);
	  }
      }
    postorder.push_back (node);
  }


  static void
  find_parents_and_children (const JoinTreeEdgeVector &edges,
			     JoinTree::ParentMap &parent_map,
			     JoinTree::ChildrenMap &children_map,
			     JoinTree::VisitOrder &postorder,
			     CliqueVector::const_iterator root)
  {
    JoinTreeVisitedNodeMap visited;
    find_parents_and_children_1 (edges, parent_map, children_map, visited,
				 postorder, root);
  }
			     

  JoinTree::
  JoinTree (const CliqueVector &cliques,
	    const std::vector<std::pair<Clique, Clique> > &clique_pairs)
    : nodes (cliques),
      edges (),
      parent_map (),
      children_map (),
      leaves (),
      postorder ()
  {
    for (std::vector<std::pair<Clique, Clique> >::const_iterator
	   ci (clique_pairs.begin ()); ci != clique_pairs.end (); ++ci)
      for (CliqueVector::const_iterator mi (nodes.begin ()); mi != nodes.end ();
	   ++mi)
	for (CliqueVector::const_iterator ni (mi + 1); ni != nodes.end (); ++ni)
	  {
	    if (ci->first == *mi && ci->second == *ni)
	      edges.push_back (JoinTreeEdge (mi, ni));
	    else if (ci->first == *ni && ci->second == *mi)
	      edges.push_back (JoinTreeEdge (ni, mi));
	  }
    assert (edge_count () == node_count () - 1);
    find_parents_and_children (edges, parent_map, children_map,
			       postorder, get_root ());

    // Find leaves.
    for (CliqueVector::const_iterator ni (nodes.begin ()); ni != nodes.end ();
	 ++ni)
      if (children_map.find (ni) == children_map.end ())
	leaves.push_back (ni);
    assert (!leaves.empty ());

  }
  
  bool
  Instance::
  is_edge (const Variable *m, const Variable *n) const
  {
    assert (m != NULL);
    assert (n != NULL);
    return (constraint_map.find (Scope (const_cast<Variable *> (m),
					const_cast<Variable *> (n)))
	    != constraint_map.end ());
  }


  bool
  Instance::
  is_empty_edge (const Variable *m, const Variable *n) const
  {
    assert (m != NULL);
    assert (n != NULL);
    assert (is_edge (m, n));
    return (constraint_map.find (Scope (const_cast<Variable *> (m),
					const_cast<Variable *> (n)))
	    ->second->empty_p ());
  }


  // Triangulate the graph with the Min-Fill algorithm.  This also
  // orders the variables of this instance accordingly (a pefect
  // elimination order).  The ordering performed in reverse of that
  // given by Dechter in Consistent Processing (since it will not
  // necessarily lead to a perfect elimination ordering), but the same
  // as Vibhav Gogate and Rina Dechter's "A Complete Anytime Algorithm
  // for Treewidth."
  VariableVector
  Instance::
  perform_min_fill (bool verbose)
  {
    const clock_t begin (std::clock ());
    std::vector<Edge> added_edges;
    unsigned int added_edges_count (0);
    VariableVector peo (variables);

    // Keep track of removed constraints (edges of the constraint
    // graph).
    std::map<Scope, Constraint *> local_constr_map (constraint_map);

    for (VariableVector::iterator ji (peo.begin ()); ji != peo.end (); ++ji)
      {
	assert (*ji != NULL);

	typedef std::map<VariableVector::iterator, size_t> Fills;
	Fills fills;

	// Find the number of fills needed for each unordered variable
	// V.
	for (VariableVector::iterator vi (ji); vi != peo.end (); ++vi)
	  {
	    assert (*vi != NULL);

	    // Zero the number of fills needed for V.
	    fills[vi] = 0;

	    // Find the number of fills needed for the parents of V.
	    for (Neighbors::const_iterator
		   mi ((*vi)->get_neighbors ().begin ());
		 mi != (*vi)->get_neighbors ().end (); ++mi)
	      {
		for (Neighbors::const_iterator
		       ni ((*vi)->get_neighbors ().begin ());
		     ni != (*vi)->get_neighbors ().end (); ++ni)
		  {
		    // These casts are safe since they are only used
		    // for constraint lookup.
		    Variable *m = const_cast<Variable *> (*mi);
		    Variable *n = const_cast<Variable *> (*ni);
		    assert (m != NULL);
		    assert (n != NULL);

		    if (m < n
			&& (local_constr_map.find (Scope (*vi, m))
			    != local_constr_map.end ())
			&& (local_constr_map.find (Scope (*vi, n))
			    != local_constr_map.end ())
			&& (local_constr_map.find (Scope (m, n))
			    == local_constr_map.end ()))
		      ++fills[vi];
		  }
	      }
	  }

	// Find variable R such that R has the smallest fill edges for
	// its parents.
	assert (fills.begin () != fills.end ());
	Fills::iterator min_fill (fills.begin ());
	for (Fills::iterator fi (fills.begin ());
	     fi != fills.end (); ++fi)
	  {
	    if (fi->second < min_fill->second)
	      min_fill = fi;
	  }
	VariableVector::iterator ri (min_fill->first);

	// Fill unfilled the parents of R with an explicit universal
	// constraint.
	for (Neighbors::iterator
	       mi ((*ri)->get_neighbors ().begin ());
	     mi != (*ri)->get_neighbors ().end (); ++mi)
	  {
	    for (Neighbors::iterator
		   ni ((*ri)->get_neighbors ().begin ());
		 ni != (*ri)->get_neighbors ().end (); ++ni)
	      {
		Variable *m = const_cast<Variable *> (*mi);
		Variable *n = const_cast<Variable *> (*ni);
		assert (m != NULL);
		assert (n != NULL);
		
		if (m < n
		    && (local_constr_map.find (Scope (*ri, m))
			!= local_constr_map.end ())
		    && (local_constr_map.find (Scope (*ri, n))
			!= local_constr_map.end ())
		    && (local_constr_map.find (Scope (m, n))
			== local_constr_map.end ()))
		  {
		    Relation *rel = new Relation (2, false, "empty_relation");
		    Constraint *univ_constr
		      = new ConflictConstraint (2, rel, "expl_univ_constraint");
		    univ_constr->add_variable (m);
		    univ_constr->add_variable (n);
		    add_constraint (univ_constr);
		    local_constr_map.insert (std::make_pair (Scope (m, n),
							     univ_constr));

		    added_edges.push_back (Edge (m, n));
		    ++added_edges_count;
		  }
	      }
	  }

	// Remove all edges incident upon R from the local map.
	for (Neighbors::iterator ni ((*ri)->get_neighbors ().begin ());
	     ni != (*ri)->get_neighbors ().end (); ++ni)
	  {
	    Variable *n = const_cast<Variable *> (*ni);
	    assert (n != NULL);
	    
	    if (local_constr_map.find (Scope (*ri, n))
		!= local_constr_map.end ())
	      local_constr_map.erase (Scope (*ri, n));
	    assert (local_constr_map.find (Scope (*ri, n))
		    == local_constr_map.end ());
	  }

	std::swap (*ri, *ji);
      }

    const clock_t end (std::clock ());

    std::cout << std::left << "Min-Fill:      "
	      << "EDGES: "
	      << std::setw (6) << added_edges_count
	      << "                       CPU: "
	      << std::setw (6) << std::setprecision (6)
	      << ((double) (end - begin)) / CLOCKS_PER_SEC
	      << std::endl;
    if (verbose)
      {
	std::cout << "  order: " << peo << std::endl;
	std::cout << "  new edges:" << std::endl;
	for (std::vector<Edge>::const_iterator ei (added_edges.begin ());
	     ei != added_edges.end (); ++ei)
	  std::cout << "    " << *ei << std::endl;
      }
    
    return peo;
  }


  void
  Instance::
  complete_paths_of_length_two (bool verbose)
  {
    const clock_t begin (std::clock ());
    std::set<SetEdge> new_edges;

    for (VariableVector::iterator i (variables.begin ());
	 i != variables.end (); ++i)
      for (VariableVector::iterator j (variables.begin ());
	   j != variables.end (); ++j)
	for (VariableVector::iterator k (variables.begin ());
	     k != variables.end (); ++k)
	  {
	    if (i != j && j != k && i != k
		&& is_edge (*i, *j) && is_edge (*j, *k) && !is_edge (*i, *k))
	      new_edges.insert (SetEdge (*i, *k));
	  }

    for (std::set<SetEdge>::const_iterator ei (new_edges.begin ());
	 ei != new_edges.end (); ++ei)
      {
	Relation *rel = new Relation (2, false, "empty_relation");
	Constraint *univ_constr
	  = new ConflictConstraint (2, rel, "expl_univ_constraint");
	Variable *i, *k;
	ei->get_variables (i, k);
	univ_constr->add_variable (i);
	univ_constr->add_variable (k);
	add_constraint (univ_constr);
	assert (is_edge (i, k));
      }

    const clock_t end (std::clock ());
    std::cout << std::left << "Compl-2paths:  "
	      << "EDGES: "
	      << std::setw (6) << new_edges.size ()
	      << "                       CPU: "
	      << std::setw (6) << std::setprecision (6)
	      << ((double) (end - begin)) / CLOCKS_PER_SEC
	      << std::endl;

    if (verbose)
      {
	std::cout << "  new edges:" << std::endl;
	for (std::set<SetEdge>::const_iterator ei (new_edges.begin ());
	     ei != new_edges.end (); ++ei)
	  std::cout << "    " << *ei << std::endl;
      }
  }


  // The variables must be in a perfect elimination ordering when
  // calling this.
  CliqueVector
  Instance::
  cliques (const VariableVector &peo, bool verbose)
  {
    const clock_t begin (std::clock ());
    CliqueVector cliques;
    unsigned int clique_count (0);
    VariableVector::size_type chromatic_num (1);

    // The size of the largest set that....
    std::map<VariableVector::const_iterator, VariableVector::size_type> size;
    
    // Initialize SIZE.
    for (VariableVector::const_iterator vi (peo.begin ()); vi != peo.end ();
	 ++vi)
      size[vi] = 0;
  
    for (VariableVector::const_iterator vi (peo.begin ()); vi != peo.end ();
	 ++vi)
      {
	std::set<VariableVector::const_iterator> clique;
	Neighbors neighbors ((*vi)->get_neighbors ());

	for (Neighbors::const_iterator ni (neighbors.begin ());
	     ni != neighbors.end (); ++ni)
	  {
	    // Find the neighbor in PEO.
	    VariableVector::const_iterator xi (std::find (peo.begin (),
							  peo.end (), *ni));
	    assert (xi != peo.end ());

	    // Add the neighbor iff it comes after V in the ordering.
	    if (vi < xi)
	      clique.insert (xi);
	  }

	if (neighbors.empty ())
	  {
	    Clique c;
	    c.insert (*vi);
	    cliques.push_back (c);
	    ++clique_count;
	  }
	
	if (clique.empty ())
	  break;

	VariableVector::const_iterator ui (*clique.begin ());
	size[ui] = std::max (size[ui], clique.size () - 1);

	if (size[vi] < clique.size ())
	  {
	    clique.insert (vi);
	    Clique c;
	    for (std::set<VariableVector::const_iterator>::const_iterator
		   ci (clique.begin ()); ci != clique.end (); ++ci)
	      c.insert (**ci);
	    cliques.push_back (c);
	    chromatic_num = std::max (chromatic_num, 1 + clique.size ());
	    ++clique_count;
	  }
      }

    clock_t end (std::clock ());

    std::cout << std::left << "Cliques:       "
	      << "CLIQUES: "
	      << std::setw (6) << clique_count
	      << "CHROMATIC: "
	      << std::setw (6) << chromatic_num
	      << "    CPU: "
	      << std::setw (6) << std::setprecision (6)
	      << ((double) (end - begin)) / CLOCKS_PER_SEC
	      << std::endl;
    if (verbose)
      {
	std::cout << "  cliques: " << std::endl;
	for (CliqueVector::const_iterator cvi (cliques.begin ());
	     cvi != cliques.end (); ++cvi)
	  std::cout << "    " << *cvi << std::endl;
      }

    return cliques;
  }


  JoinTree
  Instance::
  join_tree (const CliqueVector &cliques, bool verbose)
  {
    const clock_t begin (std::clock ());
    std::vector<std::pair<Clique, Clique> > edges;

    assert (cliques.size () > 0);
    for (CliqueVector::const_reverse_iterator mi (cliques.rbegin ());
	 mi != cliques.rend () - 1; ++mi)
      {
	VariableSet::size_type largest_shared (0);
	CliqueVector::const_reverse_iterator parent_of_m (cliques.rend () - 1);
	for (CliqueVector::const_reverse_iterator ni (mi + 1);
	     ni != cliques.rend (); ++ni)
	  {
	    VariableSet intersection;
	    std::insert_iterator<VariableSet>
	      intersection_inserter (intersection, intersection.begin ());
	    std::set_intersection (mi->begin (), mi->end (),
				   ni->begin (), ni->end (),
				   intersection_inserter);
	    if (intersection.size () > largest_shared)
	      {
		largest_shared = intersection.size ();
		parent_of_m = ni;
	      }
	  }
	// assert (largest_shared > 0);
	assert (parent_of_m != cliques.rend ());
	std::pair<Clique, Clique> edge (*parent_of_m, *mi);
	assert (std::find (edges.begin (), edges.end (), edge)
		== edges.end ());
	edges.push_back (edge);
      }

    JoinTree tree (cliques, edges);
    clock_t end (std::clock ());

    std::cout << std::left << "Join-Tree:     "
	      << "CLIQUES: "
	      << std::setw (6) << tree.node_count ()
	      << "EDGES: "
	      << std::setw (6) << tree.edge_count ()
	      << "        CPU: "
	      << std::setw (6) << std::setprecision (6)
	      << ((double) (end - begin)) / CLOCKS_PER_SEC
	      << std::endl;

    if (verbose)
      {
	std::cout << "  root: " << *tree.get_root () << std::endl;
	std::cout << "  edges:" << std::endl;
	for (std::vector<std::pair<Clique, Clique> >::const_iterator
	       ei (edges.begin ()); ei != edges.end (); ++ei)
	  {
	    std::cout << "    (" << ei->first << ", " << ei->second << ')'
		      << std::endl;
	  }
	std::cout << "  leaves:" << std::endl;
	for (std::vector<CliqueVector::const_iterator>::const_iterator
	       li (tree.get_leaves ().begin ());
	     li != tree.get_leaves ().end (); ++li)
	  std::cout << "    " << **li << std::endl;
      }

    return tree;
  }


  bool
  Instance::
  tppc (std::queue<SetTriangle> &tri_queue, EnqueuedTriagnleMap &enq_map,
	const RelatedTrianglesMap &related_triangles, PcStats *stats)
  {
    // Iterate until a fixed-point is reached.
    while (!tri_queue.empty ())
      {
	std::queue<SetEdge> edge_queue;

	// Get the triangle from the front of the queue.
	const SetTriangle &triangle (tri_queue.front ());

	// std::cerr << "Popping " << triangle << std::endl;

	// Get the variables of the triangle.
	Variable *i (NULL), *j (NULL), *k (NULL);
	triangle.get_variables (i, j, k);

	// Revise the constraint on {I, J} with respect to K.
	const bool ij_changed (revise_3 (i, j, k, stats));
	if (is_empty_edge (i, j))
	  return false;

	// Revise the constraint on {I, K} with respect to J.
	const bool ik_changed (revise_3 (i, k, j, stats));
	if (is_empty_edge (i, k))
	  return false;

	// Revise the constraint on {J, K} with respect to I.
	const bool jk_changed (revise_3 (j, k, i, stats));
	if (is_empty_edge (j, k))
	  return false;

	// If a revised constraint changed, push it onto EDGE_QUEUE.
	if (ij_changed)
	  edge_queue.push (SetEdge (i, j));
	if (ik_changed)
	  edge_queue.push (SetEdge (i, k));
	if (jk_changed)
	  edge_queue.push (SetEdge (j, k));

	while (!edge_queue.empty ())
	  {
	    const SetEdge &edge (edge_queue.front ());
	    Variable *m (NULL), *n (NULL);
	    edge.get_variables (m, n);
	    edge_queue.pop ();

	    assert (related_triangles.find (SetEdge (m, n))
		    != related_triangles.end ());
	    const SetTriangleVector
	      &related (related_triangles.find (SetEdge (m, n))->second);
	    for (SetTriangleVector::const_iterator ri (related.begin ());
		 ri != related.end (); ++ri)
	      {
		// std::cerr << "Related: " << *ri << std::endl;
		assert (enq_map.find (*ri) != enq_map.end ());
		if (!enq_map[*ri])
		  {
		    // std::cerr << "Pushing " << *ri << std::endl;
		    tri_queue.push (*ri);
		    enq_map[*ri] = true;
		  }
	      }
	  }

	// Mark this triangle dequeued and pop the triangle queue.
	assert (enq_map.find (triangle) != enq_map.end ());
	enq_map[triangle] = false;
	tri_queue.pop ();
      }

    return true;
  }


  // Perform Path-Consistency-2 (PC-2) on this instance.  Return true
  // iff this instance is path consistent.  If this instance is found
  // not to be path consistent, then some triangles may be left
  // unprocessed.
  bool
  Instance::
  perform_path_consistency_2 (PcStats *stats)
  {
    if (stats)
      stats->begin = std::clock ();

    // A queue of ((i,j), k).
    std::deque<IteratorTriple> queue;
    
    // Initialize queue with every triangle.
    for (VariableVector::iterator i (variables.begin ()); i != variables.end ();
	 ++i)
      for (VariableVector::iterator j (i + 1); j != variables.end (); ++j)
	for (VariableVector::iterator k (variables.begin ());
	     k != variables.end (); ++k)
	  {
	    assert (i < j);
	    if (k != i && k != j)
	      queue.push_back (IteratorTriple (i, k, j));
	  }

    // Iterate until a fixed-point is reached.
    while (!queue.empty ())
      {
	const IteratorTriple &front (queue.front ());
	const VariableVector::iterator i (front.first);
	const VariableVector::iterator k (front.second);
	const VariableVector::iterator j (front.third);
	queue.pop_front ();

	// Revise the constraint on {I, J} with respect to K.
	const bool changed (revise_3 (*i, *j, *k, stats));

	// If the constraint on {I, J} is empty, then this instance is
	// not path consistent.  Note that universal constraints are
	// not explicit, even at this point, so we check this only if
	// the constraint is explicit.  Furthermore, a constraint will
	// not be universal if an inconsistency was found on {I, J}.
	if (is_edge (*i, *j) && is_empty_edge (*i, *j))
	  return pc_false (stats);

	// If revise_3 modified the constraint on {I, J}, then push
	// any triangles that might need reprocessed back on the
	// queue.
	if (changed)
	  {
	    for (VariableVector::iterator l (variables.begin ());
		 l != variables.end (); ++l)
	      if (l != i && l != j)
		{
 		  queue.push_back (IteratorTriple (l, i, j));
		  queue.push_back (IteratorTriple (l, j, i));
		}
	  }
      }

    // If we made it this far, then this instance is path consistent.
    return pc_true (stats);
  }


  bool
  Instance::
  perform_directional_path_consistency (PcStats *stats)
  {
    if (stats)
      stats->begin = std::clock ();

    // Note: reverse iteration.
    for (VariableVector::reverse_iterator k (variables.rbegin ());
	 k != variables.rend (); ++k)
      for (VariableVector::reverse_iterator j (k + 1); j != variables.rend ();
	   ++j)
	for (VariableVector::reverse_iterator i (j + 1); i != variables.rend ();
	     ++i)
	  {
	    assert (i > j && k < i && k < j);
	    if (is_edge (*i, *k) && is_edge (*j, *k))
	      {
		// Revise the constraint on {I, J} with respect to K.
		revise_3 (*i, *j, *k, stats);

		// If the constraint on {I, J} is still universal,
		// make it explicit.  This may be needed above by a
		// subsequent iteration since this method works on the
		// explicit constraints alone.
		if (!is_edge (*i, *j))
		  {
		    Relation *rel = new Relation (2, false, "empty_relation");
		    Constraint *univ_constr
		      = new ConflictConstraint (2, rel, "expl_univ_constraint");
		    univ_constr->add_variable (*i);
		    univ_constr->add_variable (*j);
		    add_constraint (univ_constr);
		  }
		assert (is_edge (*i, *j));

		// If the constraint on {I, J} is empty, then this
		// instance in not path consistent.
		if (is_empty_edge (*i, *j))
		  return pc_false (stats);
	      }
	  }
    
    // If we made it this far, then this instance is directional path
    // consistent.  This does not mean that this instance is
    // necessarily path consistent, so we don't set PC-p to PC_P_YES
    // like we do with perform_path_consistency_2.
    return pc_maybe (stats);
  }

  typedef std::vector<VariableTriple> VariableTripleVector;
  typedef std::map<SetEdge, VariableTripleVector> RelatedTriplesMap;

  // Perform Partial Path-Consistency (PPC) on this instance.  Return
  // true iff this instance is partially path consistent.  If this
  // instance is found not to be path consistent, then some triangles
  // may be left unprocessed.  Based on PPC in Christian Bliek and
  // Djamila Sam-Haroud's "Path Consistency on Triangulated Constraint
  // Graphs" in Proceedings of the 16th IJCAI (p. 459).
  bool
  Instance::
  perform_partial_path_consistency (PcStats *stats)
  {
    if (stats)
      stats->begin = std::clock ();

    std::queue<SetEdge> queue;
    RelatedTriplesMap related_triples;

    // Initialize queue with every constraint (every edge in the
    // constraint graph).
    for (std::map<Scope, Constraint *>::const_iterator
	   mi (constraint_map.begin ()); mi != constraint_map.end (); ++mi)
      {
	assert (mi->second != NULL);

	// All constraints must be binary at this point.
	assert (mi->second->binary_p ());

	// Extraxt I and J from the scope of the constraint.
	const Scope &scope (mi->first);
	assert (scope.size () == 2);
	Scope::const_iterator si (scope.begin ());
	Variable *i (*si++);
	Variable *j (*si);
	assert (++si == scope.end ());
	assert (i != j);
	SetEdge e (i, j);

	queue.push (e);
	// std::cerr << "Initializing " << e << std::endl;

	for (VariableVector::iterator ki (variables.begin ());
	     ki != variables.end (); ++ki)
	  {
	    Variable *k (*ki);
	    
	    if (k != i && k != j && is_edge (i, k) && is_edge (j, k))
	      {
		related_triples[e].push_back (VariableTriple (i, j, k));
		// std::cerr << "Related: " << VariableTriple (i, j, k)
		// 	  << std::endl;
		related_triples[e].push_back (VariableTriple (i, k, j));
		// std::cerr << "Related: " << VariableTriple (i, k, j)
		// 	  << std::endl;
		related_triples[e].push_back (VariableTriple (j, k, i));
		// std::cerr << "Related: " << VariableTriple (j, k, i)
		// 	  << std::endl;
	      }
	  }
      }

    // Iterate until a fixed-point is reached.
    while (!queue.empty ())
      {
	const SetEdge &front (queue.front ());
	// std::cerr << "Popping " << front << std::endl;

	VariableTripleVector &related (related_triples[front]);
	for (VariableTripleVector::iterator tvi (related.begin ());
	     tvi != related.end (); ++tvi)
	  {
	    // std::cerr << "Related: " << *tvi << std::endl;
	    VariableTriple &triple (*tvi);
	    Variable *i (triple.first);
	    Variable *j (triple.second);
	    Variable *k (triple.third);
	    
	    // Revise the constraint on {I, J} with respect to K.
	    const bool changed (revise_3 (i, j, k, stats));
	    
	    // If the constraint on {I, J} is empty, then this
	    // instance in not path consistent.
	    if (is_empty_edge (i, j))
	      return pc_false (stats);
	    
	    // If revise_3 modified the constraint on {I, J}, then
	    // push the edge (I,J) back on the queue for
	    // reprocessing.
	    if (changed)
	      {
		// std::cerr << "Pushing " << Edge (i, j) << std::endl;
		queue.push (SetEdge (i, j));
	      }
	  }

	queue.pop ();
      }
    
    // If we made it this far, then this instance is partially path
    // consistent.
    return pc_maybe (stats);
  }


  bool
  Instance::
  perform_triangle_partial_path_consistency (PcStats *stats)
  {
    if (stats)
      stats->begin = std::clock ();

    std::queue<SetTriangle> tri_queue;
    EnqueuedTriagnleMap enq_map;
    RelatedTrianglesMap related_triangles;

    // Find each triangle.
    for (VariableVector::const_iterator i (variables.begin ());
	 i != variables.end (); ++i)
      for (VariableVector::const_iterator j (i + 1); j != variables.end (); ++j)
	for (VariableVector::const_iterator k (j + 1); k != variables.end ();
	     ++k)
	  {
	    assert (i < j);
	    assert (j < k);
	    if (is_edge (*i, *j) && is_edge (*i, *k) && is_edge (*j, *k))
	      {
		SetTriangle triangle (*i, *j, *k);

		// Enqueue this triagle.
		tri_queue.push (triangle);

		// std::cerr << "Initializing " << triangle << std::endl;

		// Mark this triangle is enqueued.
		assert (enq_map.find (triangle) == enq_map.end ());
		enq_map[triangle] = true;

		// Indicate that this triangle is related to each edge
		// of the triangle.
		related_triangles[SetEdge (*i, *j)].push_back (triangle);
		related_triangles[SetEdge (*i, *k)].push_back (triangle);
		related_triangles[SetEdge (*j, *k)].push_back (triangle);
	      }
	  }
    assert (tri_queue.size () == enq_map.size ());

    bool keep_going (tppc (tri_queue, enq_map, related_triangles, stats));
    if (keep_going)
      return pc_maybe (stats);
    else
      return pc_false (stats);
  }


  bool
  Instance::
  perform_triangle_partial_path_consistency_2 (const JoinTree &tree,
					       PcStats *stats)
  {
    if (stats)
      stats->begin = std::clock ();

    // Visit each clique in postoder.
    JoinTree::VisitOrder order (tree.get_postorder ());
    for (JoinTree::VisitOrder::const_iterator
    	   po (order.begin ()); po != order.end (); ++po)
      {
	std::queue<SetTriangle> tri_queue;
	EnqueuedTriagnleMap enq_map;
	RelatedTrianglesMap related_triangles;
    	const VariableVector vars ((*po)->begin (), (*po)->end ());

	// Find each triangle.
	for (VariableVector::const_iterator i (vars.begin ()); i != vars.end ();
	     ++i)
	  for (VariableVector::const_iterator j (i + 1); j != vars.end (); ++j)
	    for (VariableVector::const_iterator k (j + 1); k != vars.end ();
		 ++k)
	      {
		assert (i < j);
		assert (j < k);
		if (is_edge (*i, *j) && is_edge (*i, *k) && is_edge (*j, *k))
		  {
		    SetTriangle triangle (*i, *j, *k);
		
		    // Skip this triangle if it's already enqued.
		    if (enq_map[triangle])
		      continue;

		    // Enqueue this triagle.
		    tri_queue.push (triangle);
		    
		    // Mark this triangle is enqueued.
		    enq_map[triangle] = true;
		    
		    // Indicate that this triangle is related to
		    // each edge of the triangle.
		    related_triangles[SetEdge (*i, *j)].push_back (triangle);
		    related_triangles[SetEdge (*i, *k)].push_back (triangle);
		    related_triangles[SetEdge (*j, *k)].push_back (triangle);
		  }
	      }
	assert (tri_queue.size () == enq_map.size ());

	bool keep_going (tppc (tri_queue, enq_map, related_triangles, stats));
	if (!keep_going)
	  return pc_false (stats);
      }

    std::reverse (order.begin (), order.end ());
    for (JoinTree::VisitOrder::const_iterator
    	   po (order.begin ()); po != order.end (); ++po)
      {
	std::queue<SetTriangle> tri_queue;
	EnqueuedTriagnleMap enq_map;
	RelatedTrianglesMap related_triangles;
    	const VariableVector vars ((*po)->begin (), (*po)->end ());

	// Find each triangle.
	for (VariableVector::const_iterator i (vars.begin ()); i != vars.end ();
	     ++i)
	  for (VariableVector::const_iterator j (i + 1); j != vars.end (); ++j)
	    for (VariableVector::const_iterator k (j + 1); k != vars.end ();
		 ++k)
	      {
		assert (i < j);
		assert (j < k);
		if (is_edge (*i, *j) && is_edge (*i, *k) && is_edge (*j, *k))
		  {
		    SetTriangle triangle (*i, *j, *k);
		
		    // Skip this triangle if it's already enqued.
		    if (enq_map[triangle])
		      continue;

		    // Enqueue this triagle.
		    tri_queue.push (triangle);
		    
		    // Mark this triangle is enqueued.
		    enq_map[triangle] = true;
		    
		    // Indicate that this triangle is related to
		    // each edge of the triangle.
		    related_triangles[SetEdge (*i, *j)].push_back (triangle);
		    related_triangles[SetEdge (*i, *k)].push_back (triangle);
		    related_triangles[SetEdge (*j, *k)].push_back (triangle);
		  }
	      }
	assert (tri_queue.size () == enq_map.size ());

	bool keep_going (tppc (tri_queue, enq_map, related_triangles, stats));
	if (!keep_going)
	  return pc_false (stats);
      }

    return pc_maybe (stats);
  }


  bool
  Instance::
  perform_consistency (ConsistencyAlgo algo, ConsistencyStats *stats,
		       bool verbose)
  {
    bool keep_going;

    switch (algo)
      {
      case NC:
	{
	  NcStats *nc_stats = NULL;
	  if (stats)
	    nc_stats = new NcStats ();
	  keep_going = perform_node_consistency (nc_stats);
	  if (nc_stats)
	    {
	      std::cout << "NC:           " << *nc_stats << std::endl;
	      delete nc_stats;
	    }
	}
	break;

      case PC2:
      case DPC:
      case PPC:
      case TPPC:
      case TPPC2:
	{
	  PcStats *pc_stats = NULL;
	  if (stats)
	    pc_stats = new PcStats ();
	  switch (algo)
	    {
	    case PC2:
	      keep_going = perform_path_consistency_2 (pc_stats);
	      if (pc_stats)
		std::cout << "PC-2:         ";
	      break;
	    case DPC:
	      keep_going = perform_directional_path_consistency (pc_stats);
	      if (pc_stats)
		std::cout << "DPC:          ";
	      break;
	    case PPC:
	      perform_min_fill (verbose);
	      complete_paths_of_length_two (verbose);
	      keep_going = perform_partial_path_consistency (pc_stats);
	      if (pc_stats)
		std::cout << "PPC:          ";
	      break;
	    case TPPC:
	      perform_min_fill (verbose);
	      complete_paths_of_length_two (verbose);
	      keep_going = perform_triangle_partial_path_consistency (pc_stats);
	      if (pc_stats)
		std::cout << "TPPC:         ";
	      break;
	    case TPPC2:
	      {
		VariableVector peo (perform_min_fill (verbose));
		complete_paths_of_length_two (verbose);
		CliqueVector clique_vec (cliques (peo, verbose));
		JoinTree tree (join_tree (clique_vec, verbose));
		keep_going
		  = perform_triangle_partial_path_consistency_2 (tree,
								 pc_stats);
		if (pc_stats)
		  std::cout << "TPPC-2:       ";
	      }
	      break;
	    default:
	      assert (false);
	    }
	  if (pc_stats)
	    {
	      std::cout << *pc_stats << std::endl;
	      delete pc_stats;
	    }
	}
	break;

      default:
	assert (false);
      }

    return keep_going;
  }


  bool
  Instance::
  perform_consistency (std::queue<ConsistencyAlgo> algos, bool verbose)
  {
    ConsistencyStats stats;
    stats.begin = std::clock ();

    bool keep_going (true);

    while (!algos.empty () && keep_going)
      {
	ConsistencyAlgo algo (algos.front ());
	keep_going = perform_consistency (algo, &stats, verbose);
	algos.pop ();
      }

    stats.end = std::clock ();
    std::cout << "Consist:      " << stats << std::endl;
    return keep_going;
  }


  bool
  Instance::
  check (const VvpSet &vvps, unsigned int *cc) const
  {
    // Currently, this only handles binary constraints.
    assert (vvps.size () == 2);

    if (cc)
      *cc = 0;

    bool consistent (true);

    // Extract just the variables from VVPS and place them into a set.
    // This way we can easily find constraints with these variables as
    // their scopes.
    Scope scope;
    for (VvpSet::const_iterator i (vvps.begin ()); i != vvps.end (); ++i)
      {
	assert (i->first != NULL);
	// Note: this cast is safe since SCOPE is used only to find
	// matching scopes.
	scope.insert (const_cast<Variable *> (i->first));
      }

    // If this is a universal constraint, then we don't need to call
    // CHECK; it is consistent.
    std::map<Scope, Constraint *>::const_iterator
      mi (constraint_map.find (scope));
    if (mi != constraint_map.end ())
      {
	assert (mi->first == scope);
	assert (mi->second != NULL);

	consistent = mi->second->check (vvps);
	if (cc)
	  ++(*cc);
      }

    // Return CONSISTENT.  This will be true iff there are no
    // constraints with scope VARIABLES or all constraints on
    // VARIABLES are consistent.
    return consistent;
  }


  std::ostream &
  operator<< (std::ostream &out, const Edge &edge)
  {
    out << '(' << *edge.first << ',' << *edge.second << ')';
    return out;
  }


  std::ostream &
  operator<< (std::ostream &out, const ConsistencyStats &stats)
  {
    out << std::left
	<< "              "
	<< "                      "
	<< " CPU: " << std::setw (6) << std::setprecision (6)
	<< ((double) (stats.end - stats.begin)) / CLOCKS_PER_SEC;
    return out;
  }


  std::ostream &
  operator<< (std::ostream &out, const NcStats &stats)
  {
    out << std::left
	<< "              "
	<< "                      "
	<< " CPU: " << std::setw (6) << std::setprecision (6)
	<< ((double) (stats.end - stats.begin)) / CLOCKS_PER_SEC
	<< " NC-p: ";
    switch (stats.nc_p)
      {
      case NC_P_YES:
	out << "yes";
	break;
      case NC_P_NO:
	out << "no";
	break;
      case NC_P_MAYBE:
	out << "?";
	break;
      default:
	// Unreachable.
	assert (false);
      }
    return out;
  }


  std::ostream &
  operator<< (std::ostream &out, const PcStats &stats)
  {
    out << std::left
	<< " REV3: " << std::setw (6) << stats.rev3
	<< " REV: " << std::setw (6) << stats.rev
	<< " CC: " <<  std::setw (6) << stats.cc
	<< " CPU: " << std::setw (6) << std::setprecision (6)
	<< ((double) (stats.end - stats.begin)) / CLOCKS_PER_SEC
	<< " PC-p: ";
    switch (stats.pc_p)
      {
      case PC_P_YES:
	out << "yes";
	break;
      case PC_P_NO:
	out << "no";
	break;
      case PC_P_MAYBE:
	out << "?";
	break;
      default:
	// Unreachable.
	assert (false);
      }

    // These are output to stderr (for now).
    std::cout << std::endl << "  eliminated tuple counts:" << std::endl;
    for (std::map<Edge, unsigned int>::const_iterator ei
	   (stats.elim_count.begin ());
	 ei != stats.elim_count.end (); ++ei)
      {
	std::cout << "    (" << *(ei->first.first) << "," << *(ei->first.second)
		  << "): " << ei->second << std::endl;
      }

    return out;
  }


  std::ostream &
  operator<< (std::ostream &out, const Clique &clique)
  {
    out << '{';
    for (Clique::const_iterator ci (clique.begin ()); ci != clique.end ();
	 ++ci)
      {
	if (ci != clique.begin ())
	  out << ", ";
	out << **ci;
      }
    out << '}';
    return out;
  }


  // Output operator for an Instance.
  std::ostream &
  operator<< (std::ostream &out, const Instance &instance)
  {
    out << "name: `" << instance.name << "'" << std::endl
	<< "variables: " << instance.variables << std::endl;
    
    const std::vector<const Constraint *> &constraints = instance.constraints;
    for (std::vector<const Constraint *>::const_iterator
	   i (constraints.begin ());
	 i != constraints.end (); ++i)
      {
	assert (*i != NULL);
	out << **i;
      }
    return out;
  }
} // namespace CSPSolver
