// -*- C++ -*-

// CSP constraint classes.

// Chris Thiel <cthiel@cse.unl.edu>
// 2008-02-02
// CSCE 821

#ifndef Solver_Constraint_h_
#define Solver_Constraint_h_

#include "Value.h"
#include "Variable.h"
#include "XMLParser/ExpressionParser.h"
#include <cassert>
#include <map>
#include <set>
#include <string>
#include <utility>

namespace CSPSolver
{
  // The scope of a constraint.
  struct Scope : public std::set<Variable *>
  {
    Scope (void);
    Scope (Variable *var);
    Scope (Variable *first, Variable *second);
  };

  std::ostream &operator<< (std::ostream &out, const Scope &scope);

  // The type of a constraint's arity.  Note: assertions may assume
  // that this is an unsigned type.
  typedef unsigned int arity_t;

  // Forward declarations.
  class ExtensionConstraint;
  class IntensionConstraint;

  // This is an abstract base class for all constraints.
  class Constraint
  {
  public:
    Constraint (arity_t arity, const std::string &name = "");
    virtual ~Constraint (void);
    virtual void add_variable (Variable *var);
    const Scope &get_scope (void) const;
    arity_t get_arity (void) const;
    bool unary_p (void) const;
    bool binary_p (void) const;
    virtual bool empty_p (void) const = 0;

    // The check method must return true if the values given in VVPS
    // are consistent; otherwise, it must return false.  The
    // cardinality of VALUES of values must equal the arity of this
    // constraint.
    virtual bool check (const VvpSet &vvps) const = 0;

    // The merge method must merge the extension constraint EXT_CONSTR
    // into this constraint.
    virtual void merge (const ExtensionConstraint &ext_constr) = 0;
    virtual void merge (const IntensionConstraint &int_constr) = 0;
    virtual void merge (const Constraint *constr) = 0;

  protected:
    // The output_definition method must output the definition
    // (relation or predicate) to output stream OUT.
    virtual std::ostream &output_definition (std::ostream &out) const = 0;

  private:
    const arity_t arity;
    const std::string name;
    Scope scope;

    // Friends:
    friend std::ostream &operator<< (std::ostream &out, 
				     const Constraint &constraint);
  };


  // A relation.  Relations are used by all extension constraints, but
  // not by intension constraints.  For the intension constraint
  // analogue, see Predicate below.
  class Relation : public ValueVectorSet
  {
  public:
    Relation (arity_t arity, bool support, const std::string &name = "");
    arity_t get_arity (void) const;
    bool support_p (void) const;
    bool conflict_p (void) const;

  private:
    const arity_t arity;
    const bool support;
    const std::string name;

    // Friends:
    friend std::ostream &operator<< (std::ostream &out, 
				     const Relation &relation);
  };


  // This is an abstract class for all extension constraints:
  // SupportConstraint and ConflictConstraint.
  class ExtensionConstraint : public Constraint
  {
  public:
    ExtensionConstraint (arity_t arity, Relation *relation,
			 const std::string &name = "");
    virtual void add_variable (Variable *var);
    const Relation *get_definition (void) const;
    void set_definition (Relation *def);
    const std::vector<const Variable *> &get_ordered_scope (void) const;
    std::vector<Value> ordered_values (const std::set<Vvp> &vvps) const;
    virtual bool check (const VvpSet &vvps) const;
    virtual void merge (const ExtensionConstraint &ext_constr);
    virtual void merge (const IntensionConstraint &int_constr);
    virtual void merge (const Constraint *constr);
    virtual std::ostream &output_definition (std::ostream &out) const;

  private:
    Relation *definition;

    // This maintains the order of the variables in the scope of this
    // constraint since SCOPE is a set.  It is used to match variables
    // to the values of DEFINITION.
    std::vector<const Variable *> ordered_scope;
  };


  // A suport constraint in extension.
  class SupportConstraint : public ExtensionConstraint
  {
  public:
    SupportConstraint (arity_t arity, Relation *relation,
		       const std::string &name = "");
    virtual bool empty_p (void) const;
  };


  // A conflict constraint in extension.
  class ConflictConstraint : public ExtensionConstraint
  {
  public:
    ConflictConstraint (arity_t arity, Relation *relation,
			const std::string &name = "");
    virtual bool empty_p (void) const;
  };
  
  
  // A prediate.  Prediates are used by all intension constraints, but
  // not by extension constraints.  For the extension constraint
  // analogue, see Relation above.
  class Predicate
  {
  public:
    Predicate (const std::string &name = "");
    void reserve_formal_param (unsigned int pos);
    unsigned int get_formal_count (void) const;
    void set_expression (CSPXMLParser::AST *expr);
    CSPXMLParser::AST *get_expression (void) const;

  private:
    const std::string name;
    unsigned int formal_count;
    CSPXMLParser::AST *expression;

    // Friends:
    friend std::ostream &operator<< (std::ostream &out, 
				     const Predicate &predicate);
  };


  // A constraint in intension.
  class IntensionConstraint : public Constraint,
			      public CSPXMLParser::VariableValuation
  {
  public:
    IntensionConstraint (arity_t arity, const Predicate *predicate,
			 const std::string &name = "");
    const Predicate *get_definition (void) const;
    void set_param_value (unsigned int pos, Value val) const;
    void set_param_variable (unsigned int pos, const Variable *variable);
    virtual bool empty_p (void) const;
    virtual bool check (const VvpSet &vvps) const;
    virtual void merge (const ExtensionConstraint &ext_constr);
    virtual void merge (const IntensionConstraint &int_constr);
    virtual void merge (const Constraint *constr);
    virtual std::ostream &output_definition (std::ostream &out) const;

    // This is for CSPXMLParser::VariableValuation.
    virtual CSPXMLParser::FunctionValue getVarValue (unsigned int id) const;

  private:
    const Predicate *const definition;

    // This is mutable since it is only used to hold temporary values
    // for calls to getVarValue.
    mutable std::vector<Value> values;

    // Map parameter positions to variables.
    std::map<int, const Variable *> effective_variable;
  };


  // Inline definitions.


  // Default scope constructor.
  inline
  Scope::Scope (void)
  {}


  // Construct a scope with the single variable VAR.
  inline
  Scope::Scope (Variable *var)
  {
    assert (var != NULL);
    insert (var);
  }


  // Construct a scope with the two variables FIRST and SECOND.
  inline
  Scope::Scope (Variable *first, Variable *second)
  {
    assert (first != NULL);
    assert (second != NULL);
    insert (first);
    insert (second);
  }


  // Construct a constraint with arity ARITY.  NAME is used for
  // debugging only; it should normally reflect the name given in the
  // input file.
  inline
  Constraint::Constraint (arity_t arity_, const std::string &name_)
    : arity (arity_),
      name (name_),
      scope ()
  {
    assert (arity > 0);
  }


  // Return the scope of this constraint.  This must not be called
  // until all variables have been added to this constraint.
  inline const Scope &
  Constraint::get_scope (void) const
  {
    assert (arity > 0);
    assert (scope.size () == arity);
    return scope;
  }


  // Return the arity of this constraint.
  inline arity_t
  Constraint::get_arity (void) const
  {
    assert (arity > 0);
    return arity;
  }


  // Return true if this is a unary constraint; otherwise return
  // false.
  inline bool
  Constraint::unary_p (void) const
  {
    assert (arity > 0);
    assert (scope.size () == arity);
    return arity == 1;
  }

  
  // Return true if this is a binary constraint; otherwise, return
  // false.
  inline bool
  Constraint::binary_p (void) const
  {
    assert (arity > 0);
    assert (scope.size () == arity);
    return arity == 2;
  }


  // Construct a relation with arity ARITY.  If SUPPORT is true, then
  // this is a support relation; otherwise it is a conflict relation.
  // NAME is used for debugging only; it should normally reflect the
  // name given in the input file.
  inline
  Relation::Relation (arity_t arity_, bool support_, const std::string &name_)
    : arity (arity_),
      support (support_),
      name (name_)
  {
    assert (arity > 0);
  }


  // Return the arity of this relation.
  inline arity_t
  Relation::get_arity (void) const
  {
    return arity;
  }


  // Return true if this is a support relation; otherwise, return
  // false.  If this returns false, then this is a conflict relation.
  inline bool 
  Relation::support_p (void) const
  {
    return support;
  }


  // Return true if this is a conflict relation; otherwise, return
  // false.  If this returns false, then this is a support relation.
  inline bool 
  Relation::conflict_p (void) const
  {
    return !support;
  }


  // Construct an extension constraint for RELATION with arity ARITY.
  // The arity of RELATION must equal ARITY.  NAME is used for
  // debugging only; it should normally reflect the name given in the
  // input file.
  inline
  ExtensionConstraint::ExtensionConstraint (arity_t arity_, Relation *relation_,
					    const std::string &name_)
    : Constraint (arity_, name_),
      definition (relation_),
      ordered_scope ()
  {
    assert (get_arity () > 0);
    assert (definition != NULL && get_arity () == definition->get_arity ());
    assert (ordered_scope.empty ());

    // Preallocate enough room in ORDERED_SCOPE for ARITY variables.
    ordered_scope.reserve (get_arity ());

    assert (get_arity () > 0);
    assert (definition != NULL && get_arity () == definition->get_arity ());
    assert (ordered_scope.empty ());
  }


  inline const Relation *
  ExtensionConstraint::get_definition (void) const
  {
    assert (definition != NULL);
    return definition;
  }


  inline void
  ExtensionConstraint::set_definition (Relation *def)
  {
    assert (def != NULL);
    definition = def;
  }


  inline const std::vector<const Variable *> &
  ExtensionConstraint::get_ordered_scope (void) const
  {
    assert (ordered_scope.size () == get_arity ());
    return ordered_scope;
  }


  // Construct a support constraint for RELATION with arity ARITY.
  // The arity of RELATION must equal ARITY, and RELATION must be a
  // support relation.  NAME and ID are used for debugging only; they
  // should normally reflect the name and id given in the input file.
  inline
  SupportConstraint::SupportConstraint (arity_t arity_, Relation *relation_,
					const std::string &name_)
    : ExtensionConstraint (arity_, relation_, name_)
  {
    assert (get_arity () > 0);
    assert (relation_ != NULL && get_arity () == relation_->get_arity ());
    assert (relation_ != NULL && relation_->support_p ());
  }

  
  inline bool
  SupportConstraint::empty_p (void) const
  {
    return get_definition ()->empty ();
  }


  // Construct a conflict constraint for RELATION with arity ARITY.
  // The arity of RELATION must equal ARITY, and RELATION must be a
  // conflict relation.  NAME and ID are used for debugging only; they
  // should normally reflect the name and id given in the input file.
  inline
  ConflictConstraint::ConflictConstraint (arity_t arity_, Relation *relation_,
					  const std::string &name_)
    : ExtensionConstraint (arity_, relation_, name_)
  {
    assert (get_arity () > 0);
    assert (relation_ != NULL && get_arity () == relation_->get_arity ());
    assert (relation_ != NULL && relation_->conflict_p ());
  }


  inline bool
  ConflictConstraint::empty_p (void) const
  {
    assert (get_arity () > 0);

    // First, find the maximum number of supports possible for this
    // constraint.
    size_t max_num_supports (1);
    for (Scope::const_iterator si (get_scope ().begin ());
	 si != get_scope ().end (); ++si)
      {
	const Variable *var (*si);
	assert (var != NULL);
	max_num_supports *= var->get_initial_domain ().size ();
      }

    return (max_num_supports - get_definition ()->size ()) == 0;
  }


  // Construct a predicate.  NAME and ID are used for debugging only;
  // they should normally reflect the name and id given in the input
  // file.
  inline
  Predicate::Predicate (const std::string &name_)
    : name (name_),
      formal_count (0),
      expression (NULL)
  {
    assert (formal_count == 0);
    assert (expression == NULL);
  }


  // Reserve a slot for a formal parameter at position POS.  This
  // method may increase the formal method count for this predicate.
  // See Predicate::get_formal_count.
  inline void 
  Predicate::reserve_formal_param (unsigned int pos)
  {
    if (pos >= formal_count)
      formal_count = pos + 1;
  }


  // Return the number of formal parameters in this predicate.  This
  // value may change as formal parameter positions are reserved.  See
  // Predicate::reserve_formal_param.
  inline unsigned int
  Predicate::get_formal_count (void) const
  {
    return formal_count;
  }


  // Set this predicate's expression.  EXPR is an abstract syntax tree
  // (AST).
  inline void
  Predicate::set_expression (CSPXMLParser::AST *expr)
  {
    expression = expr;
  }


  // Return this predicates expression as an AST.  This will return
  // NULL if the expression has not been set yet.  See
  // Predicate::set_expression.
  inline CSPXMLParser::AST *
  Predicate::get_expression (void) const
  {
    return expression;
  }

  inline
  IntensionConstraint::IntensionConstraint (arity_t arity_, 
					    const Predicate *predicate_,
					    const std::string &name_)
    : Constraint (arity_, name_),
      definition (predicate_),
      values (),
      effective_variable ()
  {
    assert (definition != NULL);

    // Resize VALUES to accomadate a value for each effective
    // parameter.
    values.resize (definition->get_formal_count ());
  }


  inline const Predicate *
  IntensionConstraint::get_definition (void) const
  {
    assert (definition != NULL);
    return definition;
  }


  inline void 
  IntensionConstraint::set_param_value (unsigned int pos, Value val) const
  {
    values[pos] = val;
  }

  inline void
  IntensionConstraint::set_param_variable (unsigned int pos, 
					   const Variable *variable)
  {
    effective_variable.insert (effective_variable.end (),
			       std::make_pair (pos, variable));
  }


  inline bool
  IntensionConstraint::empty_p (void) const
  {
    // TODO.
    assert (false);
    return false;
  }


  // This method should not be called directly.
  inline CSPXMLParser::FunctionValue
  IntensionConstraint::getVarValue (unsigned int id) const
  {
    return values.at (id);
  }
} // namespace CSPSolver

#endif	// Solver_Constraint_h_
