// CSP constraint methods and functions.

// Chris Thiel <cthiel@cse.unl.edu>
// 2008-02-24
// CSCE 821


#include "Constraint.h"
#include "Value.h"
#include "Variable.h"
#include <algorithm>
#include <cassert>
#include <iostream>
#include <iterator>


namespace CSPSolver
{
  // Constraint destructor.  RAII: Remove this constraint from the
  // list of constraints acting on each variable in its scope.
  Constraint::
  ~Constraint (void)
  {
    for (Scope::const_iterator vi (scope.begin ()); vi != scope.end (); ++vi)
      {
	assert (*vi != NULL);
	Variable *v (*vi);
	ConstraintList &constraints (v->constraints);
	constraints.erase (std::remove (constraints.begin (),
					constraints.end (), this));
	v->constraint_removed = true;
      }
  }


  // Add the variable VAR to this constraint.  Also, this constraint
  // is added to VAR's list of constraints.
  void
  Constraint::
  add_variable (Variable *var)
  {
    assert (var != NULL);
    assert (arity > 0);

    // Add VAR to SCOPE.
    scope.insert (scope.end (), var);

    // Add this constraint to VAR.
    var->constraints.push_back (this);
    var->constraint_added = true;

    assert (scope.size () <= arity);
  }


  // Add the variable VAR to this constraint.  VAR is added at the end
  // on the list of variables, so add them in the order their
  // corresponding values appear in this constraint's relation.
  void
  ExtensionConstraint::
  add_variable (Variable *var)
  {
    assert (var != NULL);
    assert (get_arity () > 0);

    // Add VAR to ORDERED_SCOPE.
    ordered_scope.push_back (var);

    // Also, call the add_variable method of the base constraint
    // class.
    Constraint::add_variable (var);

    assert (ordered_scope.size () <= get_arity ());
  }


  std::vector<Value>
  ExtensionConstraint::
  ordered_values (const std::set<Vvp> &vvps) const
  {
    std::vector<Value> ordered_vals;
    ordered_vals.reserve (vvps.size ());

    for (std::vector<const Variable *>::const_iterator
	   osi (ordered_scope.begin ()); osi != ordered_scope.end (); ++osi)
      {
	assert (*osi != NULL);

	for (std::set<Vvp>::const_iterator vvpsi (vvps.begin ());
	     vvpsi != vvps.end (); ++vvpsi)
	  {
	    assert (vvpsi->first != NULL);

	    if (vvpsi->first == *osi)
	      ordered_vals.push_back (vvpsi->second);
	  }
      }
    assert (ordered_vals.size () == vvps.size ());

    return ordered_vals;
  }


#ifndef NDEBUG
  // Return true iff .  This function is used in assertions and must
  // have no side effects.
  static bool
  values_in_domains_p (const VvpSet &vvps)
  {
    for (VvpSet::const_iterator vvpsi (vvps.begin ());
	 vvpsi != vvps.end (); ++vvpsi)
      {
	const Variable *var = vvpsi->first;
	assert (var != NULL);
	const Value val = vvpsi->second;
	const Domain &dom = var->get_initial_domain ();

	// Try to find VALUES[I] in the domain of VAR[I].
	const Domain::const_iterator domi = std::find (dom.begin (), dom.end (),
						       val);

	// If the value is not present, return false.
	if (domi == dom.end ())
	  {
	    std::cerr << "values_in_domains_p: value `" << val
		      << "' is not in the domain of `" << *var << "', which is "
		      << dom << std::endl;
	    return false;
	  }
      }
    return true;
  }
#endif


  bool
  ExtensionConstraint::
  check (const VvpSet &vvps) const
  {
    assert (vvps.size () == get_arity ());
    assert (get_arity () == get_definition ()->get_arity ());
    assert (values_in_domains_p (vvps));

    const std::vector<Value> &values (ordered_values (vvps));

    // If we find a tuple equal to VALUES and this is a support
    // constraint, then return true.  If we find a tuple equal to
    // VALUES and this is a conflict constraint, then return false.
    for (Relation::const_iterator i = get_definition ()->begin ();
	 i != get_definition ()->end (); ++i)
      {
	if (values == *i)
	  return get_definition ()->support_p ();
      }

    // If we found no tuple equal to VALUES, then return true when
    // this is a conflict constraint and false when this is a support
    // constraint.
    return get_definition ()->conflict_p ();
  }


  bool
  IntensionConstraint::
  check (const VvpSet &vvps) const
  {
    for (std::set<Vvp>::const_iterator vvpsi (vvps.begin ());
	 vvpsi != vvps.end (); ++vvpsi)
      {
	for (std::map<int, const Variable *>::const_iterator
	       j (effective_variable.begin ()); j != effective_variable.end ();
	     ++j)
	  {
	    unsigned int pos = j->first;
	    const Variable *var = j->second;
	    
	    if (vvpsi->first == var)
	      values[pos] = vvpsi->second;
	  }
      }

    // Now, evaluate the expression.
    Value val (get_definition ()->get_expression ()->value (*this));
    assert (val == 0 || val == 1);
    return val;
  }


  void
  ExtensionConstraint::
  merge (const ExtensionConstraint &ext_constr)
  {
    assert (get_arity () == ext_constr.get_arity ());
    assert (get_scope () == ext_constr.get_scope ());
    assert (get_definition () != ext_constr.get_definition ());

    const Relation *this_relation = get_definition ();
    const Relation *ext_relation = ext_constr.get_definition ();
    Relation *merge_relation = new Relation (get_arity (),
					     get_definition ()->support_p (),
					     "merge_relation");

    // If needed, reorder the tuples of EXT_CONSTR to match those of
    // this constraint.
    ValueVectorSet ordered_ext;
    if (get_ordered_scope () != ext_constr.get_ordered_scope ())
      {
	for (ValueVectorSet::const_iterator ri (ext_relation->begin ());
	     ri != ext_relation->end (); ++ri)
	  {
	    ValueVector values;
	    values.resize (get_scope ().size ());
	    for (std::vector<const Variable *>::size_type tv_index (0);
		 tv_index < get_ordered_scope ().size (); ++tv_index)
	      {
		for (std::vector<const Variable *>::size_type ov_index (0);
		     ov_index < ext_constr.get_ordered_scope ().size ();
		     ++ov_index)
		  if (get_ordered_scope ().at (tv_index)
		      == ext_constr.get_ordered_scope ().at (ov_index))
		    {
		      values[tv_index] = ri->at (ov_index);
		    }
	      }
	    ordered_ext.insert (values);
	  }
      }
    else
      {
	ordered_ext = *ext_relation;
      }
    
    // Compute the merged relation.
    if (this_relation->support_p () && ext_relation->support_p ())
      {
	// merge = supports INTERSECTION supports.
	std::insert_iterator<Relation>
	  intersection_inserter (*merge_relation, merge_relation->begin ());
	std::set_intersection (this_relation->begin (), this_relation->end (),
			       ordered_ext.begin (), ordered_ext.end (),
			       intersection_inserter);
      }
    else if (this_relation->support_p () && ext_relation->conflict_p ())
      {
    	// merge = supports - conflicts.
    	std::insert_iterator<Relation>
    	  diff_inserter (*merge_relation, merge_relation->begin ());
    	std::set_difference (this_relation->begin (), this_relation->end (),
    			     ordered_ext.begin (), ordered_ext.end (),
    			     diff_inserter);
      }
    else if (this_relation->conflict_p () && ext_relation->support_p ())
      {
    	// merge = supports - conflicts.
    	std::insert_iterator<Relation>
    	  diff_inserter (*merge_relation, merge_relation->begin ());
    	std::set_difference (ordered_ext.begin (), ordered_ext.end (),
    			     this_relation->begin (), this_relation->end (),
    			     diff_inserter);
      }
    else if (this_relation->conflict_p () && ext_relation->conflict_p ())
      {
    	// merge = conflicts UNION conflicts.
    	std::insert_iterator<Relation>
    	  union_inserter (*merge_relation, merge_relation->begin ());
    	std::set_union (this_relation->begin (), this_relation->end (),
    			ordered_ext.begin (), ordered_ext.end (),
    			union_inserter);
      }
    else
      {
	// Unreachable.
	assert (false);
      }

    set_definition (merge_relation);
  }


  void
  ExtensionConstraint::
  merge (const IntensionConstraint &/* int_constr */)
  {
    assert (false);		// TODO.
  }


  void
  ExtensionConstraint::
  merge (const Constraint *constr)
  {
    const ExtensionConstraint *ext_constr
      = dynamic_cast<const ExtensionConstraint *> (constr);
    if (ext_constr)
      {
	merge (*ext_constr);
	return;
      }

    const IntensionConstraint *int_constr
      = dynamic_cast<const IntensionConstraint *> (constr);
    if (int_constr)
      {
	merge (int_constr);
	return;
      }

    std::cerr << typeid (constr).name () << std::endl;
    // Unreachable.
    assert (false);
  }


  void
  IntensionConstraint::
  merge (const ExtensionConstraint &/* ext_constr */)
  {
    assert (false);		// TODO.
  }


  void
  IntensionConstraint::
  merge (const IntensionConstraint &/* int_constr */)
  {
    assert (false);		// TODO.
  }


  void
  IntensionConstraint::
  merge (const Constraint */* constr */)
  {
    assert (false);		// TODO.
  }


  // An output operator for scopes.
  std::ostream &
  operator<< (std::ostream &out, const Scope &scope)
  {
    out << '{';
    for (Scope::const_iterator i (scope.begin ()); i != scope.end (); ++i)
      {
	if (i != scope.begin ())
	  out << ", ";
	assert (*i != NULL);
	out << **i;
      }
    out << '}';
    return out;
  }


  // An output operator for constraints.
  std::ostream &
  operator<< (std::ostream &out, const Constraint &constraint)
  {
    out << constraint.name << ": " << std::endl
	<< "  scope: " << constraint.scope << std::endl << "  ";
    constraint.output_definition (out) << std::endl;
    return out;
  }
  

  // An output operator for relations.
  std::ostream &
  operator<< (std::ostream &out, const Relation &relation)
  {
    out << relation.name << ": ";
    if (relation.support)
      out << "supports = ";
    else
      out << "conflicts = ";

    out << '{';
    for (ValueVectorSet::const_iterator i = relation.begin (); 
	 i != relation.end (); ++i)
      {
	if (i != relation.begin ())
	  out << ", ";
	out << *i;
      }
    out << '}';
    return out;
  }


  // Output the definition of a constraint in extension.
  std::ostream &
  ExtensionConstraint::output_definition (std::ostream &out) const
  {
    assert (definition != NULL);
    return out << *definition;
  }


  // An output operator for predicates.
  std::ostream &
  operator<< (std::ostream &out, const Predicate &predicate)
  {
    out << predicate.name << ": ";
    if (predicate.expression != NULL)
      predicate.expression->expression (out);
    else
      out << "nil";
    return out;
  }

  
  // Output the definition of a constraint in intension.
  std::ostream &
  IntensionConstraint::output_definition (std::ostream &out) const
  {
    assert (definition != NULL);
    return out << *definition;
  }
} // namespace CSPSolver
