/* urbcsp.c -- generates uniform random binary constraint satisfaction
   problems */

#include "config.h"
/* This code uses asprintf, a GNU extension.  */
#define _GNU_SOURCE
#include <assert.h>
#include <errno.h>
#include <getopt.h>
#include <limits.h>
#include <math.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* Local functions.  */
static void error (const char *fmt, ...)
  __attribute__ ((format (printf, 1, 2), noreturn));

/* Global variables.  */
static char *program_name = "urbcsp";
static char *program_version = "0.2";

/* A binary scope.  */
struct scope
{
  int first, second;
};

/* Print an error message to stderr and exit.  A newline is appended
   to the end of the message.  */
static void
error (const char *fmt, ...)
{
  va_list arg;

  assert (fmt != NULL);
  assert (program_name != NULL);
  assert (program_name[0] != '\0');

  fprintf (stderr, "%s: error: ", program_name);
  va_start (arg, fmt);
  vfprintf (stderr, fmt, arg);
  va_end (arg);
  fputc ('\n', stderr);

  exit (EXIT_FAILURE);
}

/* Like error but prefixes messages with `warning' and doesn't call
   exit.  */
static void
warning (const char *fmt, ...)
{
  va_list arg;

  assert (fmt != NULL);
  assert (program_name != NULL);
  assert (program_name[0] != '\0');

  fprintf (stderr, "%s: warning: ", program_name);
  va_start (arg, fmt);
  vfprintf (stderr, fmt, arg);
  va_end (arg);
  fputc ('\n', stderr);
}

/* Like warning but prefixes messages with `info'.  */
static void
inform (const char *fmt, ...)
{
  va_list arg;

  assert (fmt != NULL);
  assert (program_name != NULL);
  assert (program_name[0] != '\0');

  fprintf (stderr, "%s: info: ", program_name);
  va_start (arg, fmt);
  vfprintf (stderr, fmt, arg);
  va_end (arg);
  fputc ('\n', stderr);
}

/* Print the program usage information to FILE.  */
static void
print_usage (FILE *file)
{
  int name_len;

  assert (file != NULL);
  assert (program_name != NULL);
  assert (program_name[0] != '\0');

  name_len = strlen (program_name);

  fprintf (file,
"Usage: %s -n VARS -a VALS -e CONSTRAINTS [-S SUPPORTS|-C CONFLICTS]\n"
"       %*c -s SEED [-i INSTANCES] [--allow-disconnected]\n"
"       %s -n VARS -a VALS -t TIGHTNESS   -p DENSITY\n"
"       %*c -s SEED [-i INSTANCES] [--allow-disconnected]\n",
	   program_name, name_len, ' ', program_name, name_len, ' ');
  
  fputs ("\nDefaults for the options are specified in brackets.\n", file);

  fputs ("\nOptions:\n"
"  -v, --variables=NUM       create NUM variables\n"
"  -a, --values=NUM          create a domain of NUM values for each variable\n"
"  -t, --tightness=RATIO     set tightness to RATIO\n"
"  -p, --density=RATIO       set the density to RATIO\n"
"  -e, --constraints=NUM     create NUM constraints\n"
"  -S, --supports=NUM        create NUM supporting tuples in each constraint\n"
"  -C, --conflicts=NUM       create NUM conflicting tuples in each constraint\n"
"  -s, --seed=VAL            use non-negative VAL as a RNG seed\n"
"  -n, --instances=NUM       output NUM instances [1]\n"
"      --allow-disconnected  allow disconnected graphs to be output\n"
"  -h, --help                print help information and exit\n"
"  -V, --version             print version information and exit\n", file);
}

/* Print the program version information to FILE.  */
static void
print_version (FILE *file)
{
  assert (file != NULL);
  assert (program_name != NULL);
  assert (program_version != NULL);
  assert (program_version != NULL);
  assert (program_version[0] != '\0');

  fprintf (file, "%s %s\n", program_name, program_version);
}

/* Return a long int from STR.  If STR cannot be completely
   represented as a long int in the range [VAL_MIN, VAL_MAX], then
   exit with an error.  */
static long int
parse_integer (const char *str, long int val_min, long int val_max)
{
  char *end;
  long int val;

  assert (str != NULL);

  errno = 0;
  val = strtol (str, &end, 0);

  /* If errno is not already set and VAL is not within the range of
     int, then set errno to ERANGE.  */
  if (errno == 0 && (val < val_min || val > val_max))
    errno = ERANGE;

  /* Check errno.  */
  if (errno != 0)
    {
      /* Handle ERANGE specially.  */
      if (errno == ERANGE)
	error ("%s for `%s'; valid range is [%ld, %ld]", strerror (errno), str,
	       val_min, val_max);
      else
	error ("%s for `%s'", strerror (errno), str);
    }

  if (end == str)
    error ("Non-numeric leading characters or no digits found in `%s'", str);
  else if (end[0] != '\0')
    error ("Non-numeric trailing characters found in `%s'", str);

  assert (end != str);
  assert (end[0] == '\0');
  assert (val >= val_min);
  assert (val <= val_max);
  return val;
}

/* Return an int from STR.  If STR cannot be completely represented as
   an int, then exit with an error.  */
static inline int
parse_int (const char *str)
{
  assert (str != NULL);
  return parse_integer (str, INT_MIN, INT_MAX);
}

/* Return a long int from STR.  If STR cannot be completely
   represented as a long int, then exit with an error.  */
static inline long int
parse_long (const char *str)
{
  assert (str != NULL);
  return parse_integer (str, LONG_MIN, LONG_MAX);
}

static inline double
parse_double (const char *str)
{
  char *end;
  double val;

  assert (str != NULL);

  errno = 0;
  val = strtod (str, &end);
  
  if (errno != 0)
    {
      if (val == 0)
	error ("input value `%s' causes underflow: %s", str, strerror (errno));
      else if (val == HUGE_VAL || val == -HUGE_VAL)
	error ("input value `%s' causes overflow: %s", str, strerror (errno));
      else
	error ("invalid input value `%s': %s", str, strerror (errno));
    }

  if (end == str)
    error ("Non-numeric leading characters or no number found in `%s'", str);
  else if (end[0] != '\0')
    error ("Non-numeric trailing characters found in `%s'", str);

  assert (end != str);
  assert (end[0] == '\0');
  return val;
}

/* This random number generator is from William H. Press, et al.,
   _Numerical Recipes in C_, Second Ed. with corrections (1994),
   p. 282.  This excellent book is available through the WWW at
   http://nr.harvard.edu/nr/bookc.html.  The specific section
   concerning ran2, Section 7.1, is in
   <http://cfatab.harvard.edu/nr/bookc/c7-1.ps>.  */

#define IM1   2147483563
#define IM2   2147483399
#define AM    (1.0/IM1)
#define IMM1  (IM1-1)
#define IA1   40014
#define IA2   40692
#define IQ1   53668
#define IQ2   52774
#define IR1   12211
#define IR2   3791
#define NTAB  32
#define NDIV  (1+IMM1/NTAB)
#define EPS   1.2e-7
#define RNMX  (1.0 - EPS)

/* Return a random floating point value between 0.0 and 1.0 exclusive.
   If idum is negative, a new series starts (and idum is made positive
   so that subsequent calls using an unchanged idum will continue in
   the same sequence). */
float
ran2 (long int *idum)
{
  int j;
  long k;
  static long idum2 = 123456789;
  static long iy = 0;
  static long iv[NTAB];
  float temp;

#ifndef NDEBUG
  {
    /* Check that we only initialize once.  */
    static unsigned int init_count = 0;
    if (*idum < 0)
      ++init_count;
    assert (init_count <= 1);
  }
#endif

  if (*idum <= 0) {                             /* initialize */
    if (-(*idum) < 1)                           /* prevent idum == 0 */
      *idum = 1;
    else
      *idum = -(*idum);                         /* make idum positive */
    idum2 = (*idum);
    for (j = NTAB + 7; j >= 0; j--) {           /* load the shuffle table */
      k = (*idum) / IQ1;
      *idum = IA1 * (*idum - k*IQ1) - k*IR1;
      if (*idum < 0)
	*idum += IM1;
      if (j < NTAB)
	iv[j] = *idum;
    }
    iy = iv[0];
  }

  k = (*idum) / IQ1;
  *idum = IA1 * (*idum - k*IQ1) - k*IR1;
  if (*idum < 0)
    *idum += IM1;
  k = idum2/IQ2;
  idum2 = IA2 * (idum2 - k*IQ2) - k*IR2;
  if (idum2 < 0)
    idum2 += IM2;
  j = iy / NDIV;
  iy = iv[j] - idum2;
  iv[j] = *idum;
  if (iy < 1)
    iy += IMM1;
  if ((temp = AM * iy) > RNMX)
    return RNMX;                                /* avoid endpoint */
  else
    return temp;
}

static int *
allocate_sequence (size_t num_elements)
{
  int *seq = (int *) malloc (num_elements * sizeof (int));
  if (seq == NULL)
    error ("out of memory");
  return seq;
}

/* Initialize the first NUM_ELEMENTS of SEQ to 0, 1, 2, ...,
   NUM_ELEMENTS - 1.  The caller must ensure that NUM_ELEMENTS is not
   greater than the number of elements in SEQ.  In all current uses,
   we initialize all elements of SEQ, i.e., NUM_ELEMENTS equals the
   sequence size.  */
static void
initialize_sequence (int *seq, size_t num_elements)
{
  size_t i;

  assert (seq != NULL);
  for (i = 0; i < num_elements; ++i)
    {
      assert (i <= INT_MAX);
      seq[i] = i;
    }
}

/* Randomize the first NUM_ELEMENTS of SEQ.  NUM_ELEMENTS must be no
   greater than the sequence size SEQ_SIZE.  */
static void
randomize_sequence (int *seq, size_t seq_size, size_t num_elements,
		    long int *idum)
{
  size_t i;

  assert (seq != NULL);
  assert (num_elements <= seq_size);

  for (i = 0; i < num_elements; ++i)
    {
      size_t r = i + (size_t) (ran2 (idum) * (seq_size - i));
      int temp = seq[r];
      seq[r] = seq[i];
      seq[i] = temp;
    }
}

static inline void
free_sequence (int *seq)
{
  assert (seq != NULL);
  free (seq);
}

static struct scope **
allocate_scopes (size_t num_elements)
{
  size_t s;

  struct scope **scope_seq
    = (struct scope **) malloc (num_elements * sizeof (struct scope *));
  if (scope_seq == NULL)
    error ("out of memory");
  
  for (s = 0; s < num_elements; ++s)
    {
      scope_seq[s] = (struct scope *) malloc (sizeof (struct scope));
      if (scope_seq[s] == NULL)
	error ("out of memory");
    }
  return scope_seq;
}

static void
initialize_scopes (struct scope **scope_seq, int nbVariables)
{
  size_t s = 0;
  int v1, v2;

  for (v1 = 0; v1 < nbVariables - 1; ++v1)
    for (v2 = v1 + 1; v2 < nbVariables; ++v2)
      {
	assert (scope_seq[s] != NULL);
	assert (v1 != v2);
	scope_seq[s]->first = v1;
	scope_seq[s]->second = v2;
	++s;
      }
}

static void
randomize_scopes (struct scope **scope_seq, size_t seq_size,
		  size_t num_elements, long int *idum)
{
  size_t i;

  assert (scope_seq != NULL);
  assert (num_elements <= seq_size);

  for (i = 0; i < num_elements; ++i)
    {
      size_t r = i + (size_t) (ran2 (idum) * (seq_size - i));
      struct scope *temp = scope_seq[r];
      assert (temp != NULL);
      assert (scope_seq[i] != NULL);
      scope_seq[r] = scope_seq[i];
      scope_seq[i] = temp;
    }
}

static void
free_scopes (struct scope **scope_seq, size_t seq_size)
{
  size_t i;

  assert (scope_seq != NULL);

  for (i = 0; i < seq_size; ++i)
    {
      assert (scope_seq[i] != NULL);
      free (scope_seq[i]);
    }
  free (scope_seq);
}

static inline void
print_xml_declaration (FILE *file)
{
  assert (file != NULL);
  fputs ("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n", file);
}

static inline void
print_instance_start_tag (FILE *file)
{
  assert (file != NULL);
  fputs ("<instance>\n", file);
}

static inline void
print_presentation_element (FILE *file, int instance_num)
{
  assert (file != NULL);
  assert (instance_num >= 0);
  fprintf (file, "  <presentation name=\"Instance%u\" maxConstraintArity=\"2\""
	   " format=\"XCSP 2.0\"/>\n", instance_num);
}

static inline void
print_domains_start_tag (FILE *file)
{
  assert (file != NULL);
  fputs ("  <domains nbDomains=\"1\">\n", file);
}

static inline void
print_domain_element (FILE *file, int nbValues)
{
  assert (file != NULL);
  assert (nbValues >= 2);
  fprintf (file, "    <domain name=\"D0\" nbValues=\"%d\">%d..%d</domain>\n",
	   nbValues, 0, nbValues - 1);
}

static inline void
print_domains_end_tag (FILE *file)
{
  assert (file != NULL);
  fprintf (file, "  </domains>\n");
}

static void
print_domains_element (FILE *file, int nbValues)
{
  assert (file != NULL);
  assert (nbValues >= 2);
  print_domains_start_tag (file);
  print_domain_element (file, nbValues);
  print_domains_end_tag (file);
}

static inline void
print_variables_start_tag (FILE *file, int nbVariables)
{
  assert (file != NULL);
  assert (nbVariables >= 2);
  fprintf (file, "  <variables nbVariables=\"%d\">\n", nbVariables);
}

static inline void
print_variable_element (FILE *file, int variable_num)
{
  assert (file != NULL);
  assert (variable_num >= 0);
  fprintf (file, "    <variable name=\"V%d\" domain=\"D0\"/>\n", variable_num);
}

static inline void
print_variables_end_tag (FILE *file)
{
  assert (file != NULL);
  fputs ("  </variables>\n", file);
}

static void
print_variables_element (FILE *file, int nbVariables)
{
  int v;

  assert (file != NULL);
  assert (nbVariables >= 2);

  print_variables_start_tag (file, nbVariables);
  for (v = 0; v < nbVariables; ++v)
    print_variable_element (file, v);
  print_variables_end_tag (file);
}

static inline void
print_relations_start_tag (FILE *file, int nbRelations)
{
  assert (file != NULL);
  assert (nbRelations >= 0);
  fprintf (file, "  <relations nbRelations=\"%d\">\n", nbRelations);
}

static inline void
print_relation_start_tag (FILE *file, int relation_num, int nbTuples,
			  int supports)
{
  assert (file != NULL);
  assert (relation_num >= 0);
  assert (nbTuples >= 0);
  fprintf (file, "    <relation name=\"R%d\" arity=\"2\" nbTuples=\"%d\""
	   " semantics=\"%s\">", relation_num, nbTuples,
	   supports ? "supports" : "conflicts");
}

static inline void
print_relation_tuple (FILE *file, int first_value, int second_value,
		      int is_first_tuple)
{
  assert (file != NULL);
  assert (first_value >= 0);
  assert (second_value >= 0);
  fprintf (file, "%s%d %d",
	   is_first_tuple ? "" : "|", first_value, second_value);
}

static inline void
print_relation_end_tag (FILE *file)
{
  assert (file != NULL);
  fputs ("</relation>\n", file);
}

/* *IDUM must be positive.  It is negated here since we want a new
   sequence of random numbers for each relation's tuples.  */
static void
print_relation_element (FILE *file, int relation_num, int nbTuples,
			int supports, int *tuple_seq, size_t seq_size,
			long int *idum, int nbValues)
{
  int t;

  assert (file != NULL);
  assert (relation_num >= 0);
  assert (nbTuples >= 0);
  assert (tuple_seq != NULL);
  assert (nbValues >= 0);
  assert ((size_t) nbValues <= seq_size);

  print_relation_start_tag (file, relation_num, nbTuples, supports);

  initialize_sequence (tuple_seq, seq_size);
  randomize_sequence (tuple_seq, seq_size, nbTuples, idum);

  for (t = 0; t < nbTuples; ++t)
    print_relation_tuple (file, tuple_seq[t] / nbValues,
			  tuple_seq[t] % nbValues, t == 0);

  print_relation_end_tag (file);
}

static inline void
print_relations_end_tag (FILE *file)
{
  assert (file != NULL);
  fputs ("  </relations>\n", file);
}

static void
print_relations_element (FILE *file, int nbRelations, int nbTuples,
 			 int supports, size_t seq_size, long int *idum,
			 int nbValues)
{
  int *tuple_seq;
  int r;

  assert (file != NULL);
  assert (nbRelations >= 0);
  assert (nbTuples >= 0);
  assert (nbValues >= 0);
  assert ((size_t) nbValues <= seq_size);

  print_relations_start_tag (file, nbRelations);

  /* Allocate a sequence for holding the SEQ_SIZE possible tuple
     values.  */
  tuple_seq = allocate_sequence (seq_size);

  for (r = 0; r < nbRelations; ++r)
    print_relation_element (file, r, nbTuples, supports, tuple_seq, seq_size,
			    idum, nbValues);

  /* Free the tuple sequence.  It is not needed outside of this
     function.  */
  free_sequence (tuple_seq);

  print_relations_end_tag (file);
}

static inline void
print_constraints_start_tag (FILE *file, int nbConstraints)
{
  assert (file != NULL);
  assert (nbConstraints >= 0);
  fprintf (file, "  <constraints nbConstraints=\"%d\">\n", nbConstraints);
}

static inline void
print_constraint_element (FILE *file, int constraint_num, int first_variable,
			  int second_variable)
{
  assert (file != NULL);
  assert (constraint_num >= 0);
  assert (first_variable >= 0);
  assert (second_variable >= 0);
  fprintf (file, "    <constraint name=\"C%d\" arity=\"2\" scope=\"V%d V%d\""
	   " reference=\"R%d\"/>\n", constraint_num, first_variable,
	   second_variable, constraint_num);
}

static inline void
print_constraints_end_tag (FILE *file)
{
  assert (file != NULL);
  fputs ("  </constraints>\n", file);
}

static void
print_constraints_element (FILE *file, int nbConstraints, size_t seq_size,
			   long int *idum, int nbVariables,
			   struct scope **scopes)
{
  struct scope **scope_seq;
  int c;

  assert (file != NULL);
  assert (nbConstraints >= 0);
  assert (*idum > 0);
  assert (nbVariables >= 0);
  assert ((size_t) nbVariables <= seq_size);

  print_constraints_start_tag (file, nbConstraints);

  /* Allocate a sequence of SEQ_SIZE scopes.  */
  scope_seq = allocate_scopes (seq_size);

  initialize_scopes (scope_seq, nbVariables);
  randomize_scopes (scope_seq, seq_size, nbConstraints, idum);

  for (c = 0; c < nbConstraints; ++c)
    {
      assert (scope_seq[c] != NULL);
      assert (scope_seq[c]->first != scope_seq[c]->second);
      print_constraint_element (file, c, scope_seq[c]->first,
				scope_seq[c]->second);
      scopes[c]->first = scope_seq[c]->first;
      scopes[c]->second = scope_seq[c]->second;
    }

  free_scopes (scope_seq, seq_size);

  print_constraints_end_tag (file);
}

static inline void
print_instance_end_tag (FILE *file)
{
  assert (file != NULL);
  fputs ("</instance>\n", file);
}

struct component;

struct node
{
  int var;
  struct node *next;
  struct component *comp;
};

struct edge
{
  struct node *first;
  struct node *second;
};

struct component
{
  struct node *list;
  struct node *last;
};

static struct component *
init_connected_components (int nbVariables)
{
  struct component *components;
  int v;

  components
    = (struct component *) malloc (nbVariables * sizeof (struct component));
  if (!components)
    error ("out of memory");

  /* This assumes that variables are 0, 1, 2, ..., n - 1.  */
  for (v = 0; v < nbVariables; ++v)
    {
      struct node *list = (struct node *) malloc (sizeof (struct node));
      if (!list)
	error ("out of memory");
      list->var = v;
      list->next = NULL;
      list->comp = &components[v];
      components[v].list = list;
      components[v].last = list;
    }

  return components;
}

static struct edge *
init_edges (struct scope **scopes, int nbConstraints,
	    struct component *components, int nbVariables)
{
  struct edge *edges;
  int s;

  edges = (struct edge *) malloc (nbConstraints * sizeof (struct edge));
  if (!edges)
    error ("out of memory");
  
  for (s = 0; s < nbConstraints; ++s)
    {
      int c;
      
      for (c = 0; c < nbVariables; ++c)
	{
	  if (components[c].list->var == scopes[s]->first)
	    edges[s].first = components[c].list;
	  if (components[c].list->var == scopes[s]->second)
	    edges[s].second = components[c].list;
	}
    }

#ifndef NDEBUG
  {
    int e;

    for (e = 0; e < nbConstraints; ++e)
      {
	assert (edges[e].first->var == scopes[e]->first);
	assert (edges[e].second->var == scopes[e]->second);
      }
  }
#endif

  return edges;
}

static void
find_connected_components (struct edge *edges, int nbConstraints)
{
  int e;
    
  for (e = 0; e < nbConstraints; ++e)
    {
      struct component *component_of_u = edges[e].first->comp;
      struct component *component_of_v = edges[e].second->comp;
	
      assert (component_of_u != NULL);
      assert (component_of_v != NULL);

      if (component_of_u != component_of_v)
	{
	  struct node *list = component_of_v->list;

	  component_of_u->last->next = list;
	  component_of_u->last = component_of_v->last;

	  while (list)
	    {
	      list->comp = component_of_u;
	      list = list->next;
	    }

	  component_of_v->list = NULL;
	  component_of_v->last = NULL;
	}
    }
}

static unsigned int
component_count (const struct component *components, int nbVariables)
{
  unsigned int num_components = 0;
  int cc;
  
  for (cc = 0; cc < nbVariables; ++cc)
    if (components[cc].list != NULL)
      ++num_components;

  return num_components;
}

/* Create a uniform binary constraint satisfaction problem with a
   specified number of variables, domain size, tightness, and number
   of constraints.  Return non-zero iff the generated constraint graph
   is connected.  */
static int
print_xcsp (FILE *file, int nbVariables, int nbValues, int nbConstraints,
	    int nbTuples, int supports, long int *seed, unsigned int instance)
{
  const int E_MAX = nbVariables * (nbVariables - 1) / 2;
  const int NBTUPLES_MAX = nbValues * nbValues;
  struct component *components;
  struct scope **scopes;
  struct edge *edges;
  unsigned int num_components = 0;

  assert (nbVariables >= 2);
  assert (nbValues >= 2);
  assert (nbConstraints >= 0);
  assert (nbConstraints <= E_MAX);
  assert (nbTuples >= 1);
  assert (nbTuples <= NBTUPLES_MAX - 1);

  components = init_connected_components (nbVariables);
  scopes = allocate_scopes (nbConstraints);

  print_xml_declaration (file);
  print_instance_start_tag (file);
  print_presentation_element (file, instance);
  print_domains_element (file, nbValues);
  print_variables_element (file, nbVariables);
  print_relations_element (file, nbConstraints, nbTuples, supports,
			   NBTUPLES_MAX, seed, nbValues);
  print_constraints_element (file, nbConstraints, E_MAX, seed, nbVariables,
			     scopes);
  print_instance_end_tag (file);

  /* Transform array of scopes into an array of edges.  */
  edges = init_edges (scopes, nbConstraints, components, nbVariables);

  /* Find the connected components and how many.  */
  find_connected_components (edges, nbConstraints);
  num_components = component_count (components, nbVariables);

  /* TODO: free components.  */
  /* TODO: free last.  */
  /* TODO: free edges.  */
  free_scopes (scopes, nbConstraints);

  assert (num_components > 0);

  return num_components == 1;
}

int
main (int argc, char* argv[])
{
  int opt;
  int nbVariables = -1, nbValues = -1, nbConstraints = -1, nbTuples = -1;
  double tightness = -1.0, p_density = -1.0;
  long int seed = -1, saved_seed;
  int supports = 0;
  int num_instances = -1;
  int allow_disconnected = 0;
  int verbose = 0;

  struct option long_options[] = 
    {
      { "variables",          1, 0, 'n' },
      { "values",             1, 0, 'a' },
      { "tightness",          1, 0, 't' },
      { "density",            1, 0, 'p' },
      { "constraints",        1, 0, 'e' },
      { "supports",           1, 0, 'S' },
      { "conflicts",          1, 0, 'C' },
      { "seed",               1, 0, 's' },
      { "instances",          1, 0, 'i' },
      { "allow-disconnected", 0, 0, '1' },
      { "verbose",            0, 0, 'v' },
      { "help",               0, 0, 'h' },
      { "version",            0, 0, 'V' },
      { 0, 0, 0, 0 }
    };
  
  while ((opt = getopt_long (argc, argv, ":n:a:t:p:e:S:C:s:i:vhV", long_options,
			     NULL)) != -1)
    {
      switch (opt)
	{
	case 'n':
	  if (nbVariables != -1)
	    error ("the number of variables cannot be specified multiple"
		   " times");
	  nbVariables = parse_int (optarg);
	  if (nbVariables < 2)
	    error ("the number of variables must be at least 2");
	  break;
	case 'a':
	  if (nbValues != -1)
	    error ("the domain size cannot be specified multiple times");
	  nbValues = parse_int (optarg);
	  if (nbValues < 2)
	    error ("the domain size must be at least 2");
	  break;
	case 't':
	  if (tightness != -1.0)
	    error ("the tightness cannot be specified multiple times");
	  tightness = parse_double (optarg);
	  if (tightness <= 0.0 || tightness >= 1.0)
	    error ("tightness ratio must be in the range (0.0, 1.0)");
	  break;
	case 'p':
	  if (p_density != -1.0)
	    error ("the density cannot be specified multiple times");
	  p_density = parse_double (optarg);
	  if (p_density <= 0.0 || p_density > 1.0)
	    error ("density ratio must be in the range (0.0, 1.0]");
	  break;
	case 'e':
	  if (nbConstraints != -1)
	    error ("the number of constraints cannot be specified multiple"
		   " times");
	  nbConstraints = parse_int (optarg);
	  if (nbConstraints < 0)
	    error ("the number of constraints must be non-negative");
	  break;
	case 'S':
	  if (nbTuples != -1)
	    error ("the number of supports or conflicts cannot be specified"
		   " multiple times");
	  nbTuples = parse_int (optarg);
	  if (nbTuples < 1)
	    error ("the number of supports must be positive");
	  supports = 1;
	  break;
	case 'C':
	  if (nbTuples != -1)
	    error ("the number of supports or conflicts cannot be specified"
		   " multiple times");
	  nbTuples = parse_int (optarg);
	  if (nbTuples < 1)
	    error ("the number of conflicts must be positive");
	  supports = 0;
	  break;
	case 's':
	  if (seed != -1)
	    error ("the seed cannot be specified multiple times");
	  seed = parse_long (optarg);
	  if (seed <= 0)
	    error ("the seed must be positive");
	  break;
	case 'i':
	  if (num_instances != -1)
	    error ("the number of instances cannot be specified multiple"
		   " times");
	  num_instances = parse_int (optarg);
	  if (num_instances < 1)
	    error ("the number of instances must be positive");
	  break;
	case '1':
	  allow_disconnected = 1;
	  break;
	case 'v':
	  ++verbose;
	  break;
	case 'h':
	  print_usage (stdout);
	  exit (EXIT_SUCCESS);
	case 'V':
	  print_version (stdout);
	  exit (EXIT_SUCCESS);
	case '?':
	  error ("unknown option `%s'", argv[optind - 1]);
	case ':':
	  error ("option `%s' requires an argument", argv[optind - 1]);
	default:
	  /* Unreachable.  */
	  assert (0);
	}
    }

  /* Check for all required options.  */
  if (nbVariables == -1)
    error ("number of variables unspecified");
  if (nbValues == -1)
    error ("domain size unspecified");
  if (seed == -1)
    error ("seed unspecified");
  if (nbConstraints != -1 || nbTuples != -1)
    {
      if (nbConstraints == -1)
	error ("number of constraints unspecified");
      if (nbTuples == -1)
	error ("number of tuples unspecified");
      if (tightness != -1 || p_density != -1.0)
	error ("invalid usage, try `%s --help'", program_name);
    }
  else if (tightness != -1.0 || p_density != -1.0)
    {
      if (tightness == -1.0)
	error ("tightness unspecified");
      if (p_density == -1.0)
	error ("density unspecified");
      if (nbConstraints != -1 || nbTuples != -1)
	error ("invalid usage, try `%s --help'", program_name);
    }
  else
    {
      error ("invalid usage, try `%s --help'", program_name);
    }

  /* If no number of instances was specified, generate 1 instance.  */
  if (num_instances == -1)
    num_instances = 1;

  /* If tightness and density given, find nbTuples and
     nbConstraints.  */
  if (tightness != -1.0)
    {
      double e_approx, e, e_max, max_tuples, approx_nogoods, num_nogoods;

      assert (p_density != -1.0);
      assert (nbConstraints == -1);
      assert (nbTuples == -1);
      assert (supports == 0);

      e_max = (double) nbVariables * ((double) nbVariables - 1.0) / 2.0;
      e_approx = e_max * p_density;
      e = round (e_approx);

      if ((verbose && e_approx != e) || verbose > 1)
	inform ("actual density (p) is %g", e / e_max);

      nbConstraints = (int) e;
      assert (e == (double) nbConstraints);

      if (verbose)
	inform ("number of constraints (e) is %d", nbConstraints);
      if (verbose > 1)
	inform ("maximum number of constraints (e_max) is %d", (int) e_max);

      /* We first approximate the number of nogoods.  This may need to
	 be adjusted since it must be an integer value.  */
      max_tuples = (double) nbValues * nbValues;
      approx_nogoods = max_tuples * tightness;
      num_nogoods = round (approx_nogoods);

      if ((verbose && approx_nogoods != num_nogoods) ||  verbose > 1)
	inform ("actual tightness (t) is %g", num_nogoods / max_tuples);

      nbTuples = (int) num_nogoods;
      assert (num_nogoods == (double) nbTuples);

      if (verbose)
	inform ("number of conflicts per constraint is %d", nbTuples);
      if (verbose > 1)
	inform ("total number of tuples per constraint is %d",
		(int) max_tuples);
    }

  /* Check for valid values (simple are checks done above).  */
  if (nbConstraints > nbVariables * (nbVariables - 1) / 2)
    error ("number of constraints must be less than or equal to %d",
	   nbVariables * (nbVariables - 1) / 2);
  if (nbConstraints < nbVariables - 1)
    {
      if (allow_disconnected)
	warning ("all constraint graphs will be disconnected");
      else
	error ("all constraint graphs will be disconnected, increase %s or"
	       " use `--allow-disconnected'",
	       p_density != -1.0 ? "density" : "number of constraints");
    }
  if (nbTuples > nbValues * nbValues - 1)
    error ("number of %s per constraint must be less than %d (for domains of"
	   " %d values)",
	   supports ? "supports" : "conflicts", nbValues * nbValues, nbValues);
  if (nbTuples <= 1)
    error ("number of %s per constraint must be greater than 1)",
	   supports ? "supports" : "conflicts");

  saved_seed = seed;
  seed = -seed;

  /* Create NUM_INSTANCES XCSP files.  */
  {
    int i;

    for (i = 0; i < num_instances; )
      {
	char *filename;
	FILE *file;
	int ret;
	int connected;

	ret = asprintf (&filename, "%d-%d-%d-%d%c-%lu-%03d.xml", nbVariables,
			nbValues, nbConstraints, nbTuples,
			supports ? 'S' : 'C', saved_seed, i);
	if (ret == -1)
	  error ("cannot create output file name: %s", strerror (errno));

	file = fopen (filename, "w");
	if (!file)
	  error ("cannot open output file `%s': %s", filename,
		 strerror (errno));

	connected = print_xcsp (file, nbVariables, nbValues, nbConstraints,
				nbTuples, supports, &seed, i);

	ret = fclose (file);
	if (ret)
	  error ("cannot close output file `%s': %s", filename,
		 strerror (errno));

	if (!connected && verbose > 2)
	  {
	    if (allow_disconnected)
	      inform ("constraint graph is disconnected");
	    else
	      inform ("constraint graph is disconnected, trying again");
	  }

	/* If the constraint graph is disconnected and we aren't
	   allowing that, then remove the file and try again.  It
	   would probably be better to check this before writing the
	   file in the first place, but that would require quite a bit
	   of work since this feature was added so late.  */
	if (connected || allow_disconnected)
	  ++i;
	else
	  remove (filename);
      }
  }

  return 0;
}
