// -*- C++ -*-

// CSP variable class.

// Chris Thiel <cthiel@cse.unl.edu>
// 2008-02-02
// CSCE 821


#ifndef _Variable_h_
#define _Variable_h_

#include "Value.h"
#include <iostream>
#include <list>
#include <string>
#include <set>
#include <vector>


namespace CSPSolver
{
  // Forward declarations.
  class Constraint;
  class Variable;
  
  
  // Variable and constraint containers, used in Variable.
  typedef std::list<const Constraint *> ConstraintList;
  typedef std::set<const Variable *> Neighbors;


  // A CSP variable.
  class Variable
  {
  public:
    Variable (const std::string &name, int id, const Domain &domain);
    const std::string &get_name (void) const;
    Domain &get_initial_domain (void);
    const Domain &get_initial_domain (void) const;
    const Neighbors &get_neighbors (void) const;

  private:
    std::string name;
    int id;

    // This variable's initial domain.
    Domain initial_domain;

    // This caches the neighbors of this variable.  These are computed
    // by get_neighbors.
    mutable Neighbors neighbors;

    // A list of constraints with this variable in their scopes.
    ConstraintList constraints;

    // This is set to true by Constraint when a new constraint has
    // been added to CONSTRAINTS.  See Constraint::add_variable.  This
    // is used by Variable::get_neighbors.
    mutable bool constraint_added;
    mutable bool constraint_removed;

    // Friends:
    friend std::ostream &operator<< (std::ostream &out, const Variable &var);

    // Make Constraint a friend so it can set CONSTRAINTS.
    friend class Constraint;
  };


  // A variable-value pair (VVP).
  class Vvp : public std::pair<const Variable *, Value>
  {
  public:
    Vvp (const Variable *var, Value val);
    const Variable *get_variable (void) const;
    Value get_value (void) const;

    // Friends:
    friend std::ostream &operator<< (std::ostream &out, const Vvp &vvp);
  };


  class VvpSet : public std::set<Vvp>
  {
  public:
    // VvpSet (void);
    VvpSet (const Vvp &vvp);
    VvpSet (const Vvp &x, const Vvp &y);
    VvpSet (const Variable *var, Value val);
    VvpSet (const Variable *x, Value a, const Variable *y, Value b);
  };


  // Variable containers used elsewhere.
  typedef std::set<Variable *> VariableSet;
  typedef std::vector<Variable *> VariableVector;


  // Inline definitions.


  inline
  Variable::Variable (const std::string &name_, int id_,
		      const Domain &domain_)
    : name (name_),
      id (id_),
      initial_domain (domain_),
      neighbors (),
      constraints (),
      constraint_added (false),
      constraint_removed (false)
  {}


  // Return the name of this variable.
  inline const std::string &
  Variable::get_name (void) const
  {
    return name;
  }


  // Return a reference to this variable's initial domain.
  inline Domain &
  Variable::get_initial_domain (void)
  {
    return initial_domain;
  }


  // Return a const reference to this variable's initial domain.
  inline const Domain &
  Variable::get_initial_domain (void) const
  {
    return initial_domain;
  }


  inline
  Vvp::Vvp (const Variable *var, Value val)
    : std::pair<const Variable *, Value> (var, val)
  {}


  // inline
  // VvpSet::VvpSet (void)
  // {}

  inline
  VvpSet::VvpSet (const Vvp &vvp)
  {
    insert (vvp);
  }


  inline
  VvpSet::VvpSet (const Vvp &x, const Vvp &y)
  {
    insert (x);
    insert (y);
  }


  inline
  VvpSet::VvpSet (const Variable *var, Value val)
  {
    insert (Vvp (var, val));
  }


  inline
  VvpSet::VvpSet (const Variable *x, Value a, const Variable *y, Value b)
  {
    insert (Vvp (x, a));
    insert (Vvp (y, b));
  }


  std::ostream &operator<< (std::ostream &out, const Variable &var);
  std::ostream &operator<< (std::ostream &out, const Neighbors &var);
  std::ostream &operator<< (std::ostream &out, const VariableVector &var);
  std::ostream &operator<< (std::ostream &out, const Vvp &vvp);
  std::ostream &operator<< (std::ostream &out, const VvpSet &vvps);
} // namespace CSPSolver



#endif	// !_Variable_h_
