// -*- C++ -*-

// Data structures for solvers.

// Chris Thiel <cthiel@cse.unl.edu>
// 2008-02-24
// CSCE 821


#ifndef Solver_Solver_h_
#define Solver_Solver_h_

#include "config.h"
#include "Instance.h"
#include "Value.h"
#include "Variable.h"
#include <cassert>
#include <ctime>
#include <deque>
#include <iomanip>
#include <iostream>
#include <set>
#include <string>


namespace CSPSolver
{
  // Forward declaration.
  class Solver;

  // A search path.
  typedef std::vector<const Variable *> SearchPath;

  // A map from variables to their current domains.
  typedef std::map<SearchPath::value_type, Domain> CurrentDomains;


  // A map from variables to their values.
  typedef std::map<SearchPath::value_type, Value> Assignments;


  // Astract base class for all CSP solvers.
  class Solver
  {
  private:
    const std::string name;

  protected:
    const Instance *problem;

    // These are used to compute the CPU time required to solve a
    // problem.
    clock_t solve_begin;
    clock_t solve_end;
    
    // The number of constraint checks.
    unsigned int cc;

    Solver (const Instance *problem, const std::string &name = "");
    virtual ~Solver (void);

  public:
    VariableVector::size_type num_vars (void) const;

    // All solvers must implement this method.  This method must find
    // the solution(s).
    virtual void solve (void) = 0;

    // All solvers must implement this method.  This method must
    // return a string with the name of the general algorithm
    // implemented by the solver (BT, CBJ, etc.).
    virtual std::string algorithm_string (void) const = 0;
  };


  // Abstract class for any backtrack search hybrid.
  class SystematicSearchSolver : public Solver
  {
    // Let bcssp access this class's members directly to enhance
    // encapsulation.
    friend void bcssp (SystematicSearchSolver &solver);

  public:
    // Variable-ordering heuristics.
    enum VariableOrdering { NO_VAR_ORDER, LEAST_DOMAIN, DEGREE,
			    DOMAIN_DEGREE_RATIO };
    
    // Value-ordering heuristics (unused for now).
    enum ValueOrdering { UNUSED };

  protected:
    // The number of nodes visited (incremented every time a value is
    // assigned to a variable).
    unsigned int nv;

    // The number of backtracks (incremented every time a consistent
    // assignment).
    unsigned int bt;

    // This is the number of solutions found so far.
    unsigned int num_solutions;

    // This is a dummy variable used in the implementation if the
    // label and unlabel methods.
    const SearchPath::value_type dummy;

  private:
    // The current domain of all variables.
    CurrentDomains current_domains;

    // The current search path.
    SearchPath current_path;

    // The assignments for all instantiated variables.
    Assignments assignments;

    // This solvers variable ordering heuristic and whether to use
    // dynamic variable ordering.
    const VariableOrdering var_order;
    const bool dynamic_var_order;

    // This solvers value ordering heuristic and whether to use
    // dynamic value ordering.
    const ValueOrdering val_order;
    const bool dynamic_val_order;

    void init_current_domains (void);
    void init_current_path (void);
    void init_assignments (void);
    void order_current_path (SearchPath::iterator first, 
			     SearchPath::iterator last);

  protected:

    // Methods for CURRENT_DOMAINS.
    const CurrentDomains::mapped_type &
    get_current_domain_at (const CurrentDomains::key_type &var) const;
    void
    remove_from_current_domain_at (const CurrentDomains::key_type &var, 
				   const CurrentDomains::mapped_type::
				   value_type &val);
    void
    set_current_domain_at (const CurrentDomains::key_type &var,
			   const CurrentDomains::mapped_type &dom);
    void
    reset_current_domain_at (const CurrentDomains::key_type &var);

    // Methods for CURRENT_PATH.
    SearchPath &get_current_path (void); // TODO: get rid of this.
    const SearchPath &get_current_path (void) const;

    // Methods for ASSIGNMENTS.
    void assign (SearchPath::iterator i, const Assignments::mapped_type &val);
    void unassign (SearchPath::const_iterator first,
		   SearchPath::const_iterator last);
    const Assignments::mapped_type &
    get_value (const Assignments::key_type &var) const;
    
    // This is true if we want to find all solutions.
    const bool find_all;

    SystematicSearchSolver (const Instance *problem, VariableOrdering var_order,
			    bool dynamic_var_order, ValueOrdering val_order,
			    bool dynamic_val_order, bool find_all = false,
			    const std::string &name = "");

  public:
    virtual ~SystematicSearchSolver (void);

    // All SystematicSearchSolvers must implement these methods as
    // described in Patrick Prosser, Hybrid Algorithms for the
    // Constraint Satisfaction Problem.  These will be called by the
    // bcssp function.
    virtual SearchPath::iterator label (SearchPath::iterator i,
					bool &consistent) = 0;
    virtual SearchPath::iterator unlabel (SearchPath::iterator i,
					  bool &consistent) = 0;

    // A SystematicSearchSolvers must implement this method.  This is
    // an unlabel method that is called when we have a solution but we
    // want to find more.
    virtual SearchPath::iterator unlabel_solution (bool &consistent) = 0;

    std::string ordering_string (void) const;
  };


  // A function to folve a binary systematic search problem as
  // described in Patrick Prosser, Hybrid Algorithms for the
  // Constraint Satisfaction Problem.  This function will call the
  // label and unlabel methods of SOLVER.
  void bcssp (SystematicSearchSolver &solver);


  // A backtrack solver.
  class BtSolver : public SystematicSearchSolver
  {
  private:
    virtual SearchPath::iterator label (SearchPath::iterator i,
					bool &consistent);
    virtual SearchPath::iterator unlabel (SearchPath::iterator i,
					  bool &consistent);
    virtual SearchPath::iterator unlabel_solution (bool &consistent);

  public:
    BtSolver (const Instance *problem, VariableOrdering var_order,
	      bool dynamic_var_order, ValueOrdering val_order,
	      bool dynamic_val_order, bool find_all = false,
	      const std::string &name = "");
    virtual void solve (void);
    virtual std::string algorithm_string (void) const;
  };


  typedef std::map<SearchPath::value_type, std::deque<Domain> > Reductions;
  typedef Reductions::mapped_type ReductionsDeque;
  
  // Note that FutureFc and PastFc are different types.
  typedef std::map<SearchPath::value_type,
		   std::deque<SearchPath::value_type> > FutureFcs;
  typedef FutureFcs::mapped_type FutureFcsDeque;
  typedef std::map<SearchPath::value_type,
		   std::deque<SearchPath::iterator> > PastFcs;
  typedef PastFcs::mapped_type PastFcsDeque;
  
  // A forward checking solver.
  class FcSolver : public virtual BtSolver
  {
  private:
    Reductions reductions;
    FutureFcs future_fcs;
    PastFcs past_fcs;

    // Datastructure initialization methods (called in the
    // constructor only).
    void init_reductions (void);
    void init_past_fcs (void);
    void init_future_fcs (void);

  protected:
    // Methods for REDUCTIONS. 

    void
    push_onto_reductions_at (const Reductions::key_type &var,
			     const Reductions::mapped_type::value_type &
			     reduction);

    const Reductions::mapped_type &
    get_reductions_at (const Reductions::key_type &var) const;

    const Reductions::mapped_type::value_type &
    top_reduction_at (const Reductions::key_type &var) const;

    void 
    pop_reduction_at (const Reductions::key_type &var);

    // Methods for FUTURE_FCS.
    void
    push_onto_future_fcs_at (const FutureFcs::key_type &var,
			     const FutureFcs::mapped_type::value_type &future);

    const FutureFcs::mapped_type &
    get_future_fcs_at (const FutureFcs::key_type &var) const;

    void
    clear_future_fcs_at (const FutureFcs::key_type &var);

    // Methods for PAST_FCS.
    void
    push_onto_past_fcs_at (const PastFcs::key_type &var,
			   const PastFcs::mapped_type::value_type &past);

    const PastFcs::mapped_type &
    get_past_fcs_at (const PastFcs::key_type &var);

    void 
    pop_past_fcs_at (const PastFcs::key_type &var);

    // Prosser's FC methods.
    bool check_forward (SearchPath::iterator i, SearchPath::iterator j);
    void undo_reductions (SearchPath::iterator i);
    void update_current_domain (SearchPath::iterator i);

  private:
    virtual SearchPath::iterator label (SearchPath::iterator i,
					bool &consistent);
    virtual SearchPath::iterator unlabel (SearchPath::iterator i,
					  bool &consistent);
    virtual SearchPath::iterator unlabel_solution (bool &consistent);

  public:
    FcSolver (const Instance *problem, VariableOrdering var_order,
	      bool dynamic_var_order, ValueOrdering val_order,
	      bool dynamic_val_order, bool find_all = false,
	      const std::string &name = "");
    virtual std::string algorithm_string (void) const;
  };


  // A set of variable vector iterators.
  typedef std::set<SearchPath::iterator> SearchPathIterSet;


  // A conflict-directed back-jumping solver.
  class CbjSolver : public virtual BtSolver
  {
    typedef std::map<SearchPath::value_type, SearchPathIterSet> ConfSets;
    typedef std::map<SearchPath::value_type, bool> BacktrackFlags;

  private:
    // Conflict sets.
    ConfSets conf_sets;
    
    void init_conf_sets (void);

  protected:
    void set_conf_set_at (const SearchPath::value_type &var,
			  const ConfSets::mapped_type &conf_set);
    void insert_into_conf_set_at (const SearchPath::value_type &var,
				  SearchPath::iterator i);
    const ConfSets::mapped_type &
    conf_set_at (const SearchPath::value_type &var) const;
    void reset_conf_set_at (const SearchPath::value_type &var);

  private:
    virtual SearchPath::iterator label (SearchPath::iterator i,
					bool &consistent);
    virtual SearchPath::iterator unlabel (SearchPath::iterator i,
					  bool &consistent);
    virtual SearchPath::iterator unlabel_solution (bool &consistent);

  public:
    CbjSolver (const Instance *problem, VariableOrdering var_order,
	       bool dynamic_var_order, ValueOrdering val_order,
	       bool dynamic_val_order, bool find_all = false,
	       const std::string &name = "");
    virtual std::string algorithm_string (void) const;
  };

  
  class CbjCbfSolver : public virtual CbjSolver
  {
    typedef std::map<SearchPath::value_type, bool> BacktrackFlags;

  private:
    // Chronological backtrack flags.
    BacktrackFlags cbfs;
    
    void init_cbfs (void);

  protected:
    bool cbf_at (const SearchPath::value_type &var) const;
    void set_all_cbfs (void);
    void zero_cbf_at (const SearchPath::value_type &var);

  private:
    virtual SearchPath::iterator label (SearchPath::iterator i,
					bool &consistent);
    virtual SearchPath::iterator unlabel (SearchPath::iterator i,
					  bool &consistent);
    virtual SearchPath::iterator unlabel_solution (bool &consistent);
    
  public:
    CbjCbfSolver (const Instance *problem, VariableOrdering var_order,
		  bool dynamic_var_order, ValueOrdering val_order,
		  bool dynamic_val_order, bool find_all = false,
		  const std::string &name = "");
    virtual std::string algorithm_string (void) const;
  };


  class FcCbjSolver : public FcSolver,
		      public CbjSolver
  {
  private:
    virtual SearchPath::iterator label (SearchPath::iterator i,
					bool &consistent);
    virtual SearchPath::iterator unlabel (SearchPath::iterator i,
					  bool &consistent);
    virtual SearchPath::iterator unlabel_solution (bool &consistent);

  public:
    FcCbjSolver (const Instance *problem, VariableOrdering var_order,
		 bool dynamic_var_order, ValueOrdering val_order,
		 bool dynamic_val_order, bool find_all = false,
		 const std::string &name = "");
    virtual std::string algorithm_string (void) const;
  };


  // Inline definitions.


  // Construct a Solver for PROBLEM.  NAME is used only for debugging.
  inline
  Solver::Solver (const Instance *problem_, const std::string &name_)
    : name (name_),
      problem (problem_),
      solve_begin (0),
      solve_end (0),
      cc (0)
  {
    assert (problem != NULL);
    assert (solve_begin == 0);
    assert (solve_end == 0);
    assert (cc == 0);
  }


  // This solver doesn't own anything.
  inline
  Solver::~Solver (void)
  {}


  // Get the number of variables in this solver's problem.
  inline VariableVector::size_type
  Solver::num_vars (void) const
  {
    return problem->get_variables ().size ();
  }


  // Construct a systematic search solver for PROBLEM.  NAME is used
  // only in debugging.  VAR_ORDER is the variable ordering heuristic,
  // and DYNAMIC_VAR_ORDER should be true if dynamic variable is to be
  // performed.  VAL_ORDER is the value ordering heuristic, and
  // DYNAMIC_VAL_ORDER should be true if dynamic value ordering is to
  // be used.
  inline
  SystematicSearchSolver::SystematicSearchSolver (const Instance *problem_,
						  VariableOrdering var_order_,
						  bool dynamic_var_order_,
						  ValueOrdering val_order_,
						  bool dynamic_val_order_,
						  bool find_all_, 
						  const std::string &name_)
    : Solver (problem_, name_),
      nv (0),
      bt (0),
      num_solutions (0),
      dummy (new Variable ("dummy", -1, Domain ())),
      current_domains (),
      current_path (),
      assignments (),
      var_order (var_order_),
      dynamic_var_order (dynamic_var_order_),
      val_order (val_order_),
      dynamic_val_order (dynamic_val_order_),
      find_all (find_all_)
  {
    assert (solve_begin == 0);
    assert (solve_end == 0);
    assert (cc == 0);
    assert (nv == 0);
    assert (bt == 0);
    assert (num_solutions == 0);
    assert (dummy != NULL);
    assert (dummy->get_name () == "dummy");
    assert (dummy->get_initial_domain ().empty ());
    assert (dummy->get_neighbors ().empty ());
    assert (current_domains.empty ());
    assert (current_path.empty ());
    assert (assignments.empty ());

    // Dynamic value ordering is not yet implemented.
    assert (!dynamic_val_order);

    // Initialize data structures.
    init_current_domains ();
    init_current_path ();
    init_assignments ();

    assert (solve_begin == 0);
    assert (solve_end == 0);
    assert (cc == 0);
    assert (nv == 0);
    assert (bt == 0);
    assert (num_solutions == 0);
    assert (dummy != NULL);
    assert (dummy->get_name () == "dummy");
    assert (dummy->get_initial_domain ().empty ());
    assert (dummy->get_neighbors ().empty ());
    assert (current_domains.size () == num_vars () + 1);
    assert (current_domains.find (dummy) != current_domains.end ());
    assert (current_domains.find (dummy)->second.empty ());
    assert (current_path.size () == num_vars () + 1);
    assert (current_path.front () == dummy);
    assert (assignments.size () == 1);
  }


  // Destroy this systematic search solver.
  inline
  SystematicSearchSolver::~SystematicSearchSolver (void)
  {
    assert (current_domains.size () == num_vars () + 1);
    assert (current_domains.find (dummy) != current_domains.end ());
    assert (current_domains.find (dummy)->second.empty ());
    assert (current_path.size () == num_vars () + 1);
    assert (current_path.front () == dummy);
    assert (assignments.size () <= num_vars () + 1);

    delete dummy;
  }


  // Construct a BT solver.  The parameters are the same as for
  // SystematicSearchSolver.
  inline
  BtSolver::BtSolver (const Instance *problem, VariableOrdering var_order,
		      bool dynamic_var_order, ValueOrdering val_order,
		      bool dynamic_val_order, bool find_all,
		      const std::string &name)
    : SystematicSearchSolver (problem, var_order, dynamic_var_order,
			      val_order, dynamic_val_order, find_all, name)
  {
    assert (solve_begin == 0);
    assert (solve_end == 0);
    assert (cc == 0);
    assert (nv == 0);
    assert (bt == 0);
    assert (num_solutions == 0);
  }


  // Construct an FC solver.  The parameters are the same as for
  // BtSolver.
  inline
  FcSolver::FcSolver (const Instance *problem, VariableOrdering var_order,
		      bool dynamic_var_order, ValueOrdering val_order,
		      bool dynamic_val_order, bool find_all,
		      const std::string &name)
    : BtSolver (problem, var_order, dynamic_var_order, val_order, 
		dynamic_val_order, find_all, name),
      reductions (),
      future_fcs (),
      past_fcs ()
  {
    assert (solve_begin == 0);
    assert (solve_end == 0);
    assert (cc == 0);
    assert (nv == 0);
    assert (bt == 0);
    assert (num_solutions == 0);
    assert (reductions.empty ());
    assert (future_fcs.empty ());
    assert (past_fcs.empty ());

    // Initialize data structures.
    init_reductions ();
    init_future_fcs ();
    init_past_fcs ();

    assert (solve_begin == 0);
    assert (solve_end == 0);
    assert (cc == 0);
    assert (nv == 0);
    assert (bt == 0);
    assert (num_solutions == 0);
    assert (reductions.size () == num_vars ());
    assert (future_fcs.size () == num_vars () + 1);
    assert (past_fcs.size () == num_vars () + 1);
  }


  // Construct a CBJ solver.  The parameters are the same as for
  // BtSolver.
  inline
  CbjSolver::CbjSolver (const Instance *problem, VariableOrdering var_order,
			bool dynamic_var_order, ValueOrdering val_order,
			bool dynamic_val_order, bool find_all,
			const std::string &name)
    : BtSolver (problem, var_order, dynamic_var_order, val_order, 
		dynamic_val_order, find_all, name),
      conf_sets ()
  {
    assert (solve_begin == 0);
    assert (solve_end == 0);
    assert (cc == 0);
    assert (nv == 0);
    assert (bt == 0);
    assert (num_solutions == 0);
    assert (conf_sets.empty ());

    // Initialize data structures.
    init_conf_sets ();
    
    assert (solve_begin == 0);
    assert (solve_end == 0);
    assert (cc == 0);
    assert (nv == 0);
    assert (bt == 0);
    assert (num_solutions == 0);
    assert (conf_sets.size () == num_vars () + 1);
  }


  inline
  CbjCbfSolver::CbjCbfSolver (const Instance *problem,
			      VariableOrdering var_order,
			      bool dynamic_var_order, ValueOrdering val_order,
			      bool dynamic_val_order, bool find_all,
			      const std::string &name)
    : BtSolver (problem, var_order, dynamic_var_order, val_order, 
		dynamic_val_order, find_all, name),
      CbjSolver (problem, var_order, dynamic_var_order, val_order, 
		 dynamic_val_order, find_all, name),
      cbfs ()
  {
    assert (solve_begin == 0);
    assert (solve_end == 0);
    assert (cc == 0);
    assert (nv == 0);
    assert (bt == 0);
    assert (num_solutions == 0);
    assert (cbfs.empty ());
    
    // Initialize data structures.
    init_cbfs ();

    assert (solve_begin == 0);
    assert (solve_end == 0);
    assert (cc == 0);
    assert (nv == 0);
    assert (bt == 0);
    assert (num_solutions == 0);
    assert (cbfs.size () == num_vars ());
  }


  inline
  FcCbjSolver::FcCbjSolver (const Instance *problem, VariableOrdering var_order,
			    bool dynamic_var_order, ValueOrdering val_order,
			    bool dynamic_val_order, bool find_all,
			    const std::string &name)
    : BtSolver (problem, var_order, dynamic_var_order, val_order, 
		dynamic_val_order, find_all, name),
      FcSolver (problem, var_order, dynamic_var_order, val_order, 
		dynamic_val_order, find_all, name),
      CbjSolver (problem, var_order, dynamic_var_order, val_order, 
		 dynamic_val_order, find_all, name)
  {
    assert (solve_begin == 0);
    assert (solve_end == 0);
    assert (cc == 0);
    assert (nv == 0);
    assert (bt == 0);
    assert (num_solutions == 0);
  }


  // Output operators.
  std::ostream &operator<< (std::ostream &out, const SearchPath &path);
  std::ostream &operator<< (std::ostream &out, const Assignments &assignments);
  std::ostream &operator<< (std::ostream &out, const CurrentDomains &domains);
  std::ostream &operator<< (std::ostream &out,
			    const SearchPathIterSet &set);
} // namespace CSPSolver

#endif	// !Solver_Solver_h_
