// Solver functions and methods.

// Chris Thiel <cthiel@cse.unl.edu>
// 2008-02-28
// CSCE 821


#include "Solver.h"
#include "Variable.h"
#include <algorithm>
#include <cassert>
#include <ctime>
#include <functional>
#include <ext/functional>	// For compose1; an SGI extension.
#include <iterator>
#include <limits>
#include <set>
#include <utility>


namespace CSPSolver
{
  // Initialize the current domains.  This must be called before
  // init_current_path.
  void
  SystematicSearchSolver::
  init_current_domains (void)
  {
    assert (solve_begin == 0);
    assert (solve_end == 0);
    assert (cc == 0);
    assert (nv == 0);
    assert (bt == 0);
    assert (num_solutions == 0);
    assert (current_domains.empty ());
    assert (current_path.empty ());
    assert (assignments.size () <= num_vars () + 1);

    // Insert the domain of DUMMY into CURRENT_DOMAINS.
    current_domains[dummy] = (dummy->get_initial_domain ());
    
    // Continue populating CURRENT_DOMAINS with the initial domains of
    // each variable in PROBLEM's variables.
    for (VariableVector::const_iterator i (problem->get_variables ().begin ());
	 i != problem->get_variables ().end (); ++i)
      {
	assert (*i != NULL);
	assert (*i != dummy);
	assert (current_domains.find (*i) == current_domains.end ());

	current_domains.insert (current_domains.end(),
			       std::make_pair (*i, 
					       (*i)->get_initial_domain ()));

	assert (current_domains.find (*i) != current_domains.end ());
      }

    assert (solve_begin == 0);
    assert (solve_end == 0);
    assert (cc == 0);
    assert (nv == 0);
    assert (bt == 0);
    assert (num_solutions == 0);
    assert (current_domains.size () == num_vars () + 1);
    assert (current_domains.find (dummy) != current_domains.end ());
    assert (current_domains.find (dummy)->second.empty ());
    assert (current_path.empty ());
    assert (assignments.size () <= num_vars () + 1);
  }


  // Initialize the current path.  This must be called after
  // init_current_domains.
  void 
  SystematicSearchSolver::
  init_current_path (void)
  {
    assert (solve_begin == 0);
    assert (solve_end == 0);
    assert (cc == 0);
    assert (nv == 0);
    assert (bt == 0);
    assert (num_solutions == 0);
    assert (current_domains.size () == num_vars () + 1);
    assert (current_domains.find (dummy) != current_domains.end ());
    assert (current_domains.find (dummy)->second.empty ());
    assert (current_path.empty ());
    assert (assignments.size () <= num_vars () + 1);

    // Preallocate enough room to hold all of the problem's variables
    // + 1 (for the dummy variable).
    current_path.reserve (num_vars () + 1);

    // Push DUMMY onto CURRENT_PATH.
    current_path.push_back (dummy);

    // Continue populating CURRENT_PATH with the PROBLEM's variables.
    current_path.insert (current_path.end (),
			 problem->get_variables ().begin (),
			 problem->get_variables ().end ());

    // Order CURRENT_PATH now.  We start at CURRENT_PATH.begin () + 1
    // because DUMMY must always be first.  Note that we order the
    // whole path even when using dynamic variable ordering, so that
    // the first real variable is ready and so we can do consistency
    // checks, which may depend on the variable ordering.
    order_current_path (current_path.begin () + 1, current_path.end ());

    assert (solve_begin == 0);
    assert (solve_end == 0);
    assert (cc == 0);
    assert (nv == 0);
    assert (bt == 0);
    assert (num_solutions == 0);
    assert (current_domains.size () == num_vars () + 1);
    assert (current_domains.find (dummy) != current_domains.end ());
    assert (current_domains.find (dummy)->second.empty ());
    assert (current_path.size () == num_vars () + 1);
    assert (current_path.front () == dummy);
    assert (assignments.size () <= num_vars () + 1);
  }


  void
  SystematicSearchSolver::
  init_assignments (void)
  {
    assert (solve_begin == 0);
    assert (solve_end == 0);
    assert (cc == 0);
    assert (nv == 0);
    assert (bt == 0);
    assert (num_solutions == 0);
    assert (current_domains.empty ()
	    || (current_domains.size () == num_vars () + 1
		&& current_domains.find (dummy) != current_domains.end ()
		&& current_domains.find (dummy)->second.empty ()));
    assert (current_path.empty ()
	    || (current_path.size () == num_vars () + 1
		&& current_path.front () == dummy));
    assert (assignments.empty ());

    // This is a dummy assignment to make thing easier later on.
    assignments[dummy] = -1;

    assert (solve_begin == 0);
    assert (solve_end == 0);
    assert (cc == 0);
    assert (nv == 0);
    assert (bt == 0);
    assert (num_solutions == 0);
    assert (current_domains.empty ()
	    || (current_domains.size () == num_vars () + 1
		&& current_domains.find (dummy) != current_domains.end ()
		&& current_domains.find (dummy)->second.empty ()));
    assert (current_path.empty ()
	    || (current_path.size () == num_vars () + 1
		&& current_path.front () == dummy));
    assert (assignments.size () == 1);
  }


  // A unary predicate that returns true iff a given variable is
  // instantiated.
  class Instantiated : public std::unary_function<const Variable *, bool>
  {
  private:
    const Assignments &assignments;

  public:
    // Construct an Instantiated predicate.  
    Instantiated (const Assignments &assignments_)
      : assignments (assignments_)
    {}


    // Return true iff VAR is instantiated.
    bool
    operator() (const Variable *var) const
    {
      return assignments.find (var) != assignments.end ();
    }
  };


  // This is the base class of all variable ordering predicates.
  struct VariableOrderPredicate
    : public std::binary_function<const Variable *, const Variable *, bool>
  {
    // Return a const reference to the name of VAR.
    const std::string &
    name (const Variable *var) const
    {
      assert (var != NULL);
      assert (!var->get_name ().empty ());

      return var->get_name ();
    }
  };


  // A binary predicate for sorting variables by least domain (LD).
  class LdPredicate : public virtual VariableOrderPredicate
  {
    typedef CurrentDomains::mapped_type::size_type DomainSizeType;

  protected:
    const CurrentDomains &current_domains;

    // Return the curent domain size of VAR.
    DomainSizeType
    domain_size (const Variable *var) const
    {
      assert (var != NULL);
      assert (current_domains.find (var) != current_domains.end ());

      return current_domains.find (var)->second.size ();
    }

  public:
    // Construct an LD predicate.
    LdPredicate (const CurrentDomains &current_domains_)
      : current_domains (current_domains_)
    {}


    // Return true iff LEFT has a smaller current domain than that of
    // RIGHT with ties being broken by lexicographical order of the
    // variables' names.
    bool 
    operator() (const Variable *left, const Variable *right) const
    {
      assert (left != NULL);
      assert (right != NULL);

      const DomainSizeType left_dom_size (domain_size (left));
      const DomainSizeType right_dom_size (domain_size (right));

      return (left_dom_size < right_dom_size
	      || (!(right_dom_size < left_dom_size)
		  && name (left) < name (right)));
    }
  };


  // A binary predicate for sorting variables by largest future degree
  // (DEG) (the largest number of unassigned neighbors).
  class DegPredicate : public virtual VariableOrderPredicate
  {
  protected:
    typedef std::iterator_traits<Neighbors::const_iterator>::difference_type
    NeighborCount;

    const Assignments &assignments;

    // Return the number of V's neighbors that are unassigned.
    NeighborCount
    unassigned_neighbors_count (const Variable *v) const
    {
      assert (v != NULL);

      return std::count_if (v->get_neighbors ().begin (),
			    v->get_neighbors ().end (),
			    __gnu_cxx::compose1 (std::logical_not<bool> (), 
						 Instantiated (assignments)));
    }

  public:
    // Construct a DEG predicate.
    DegPredicate (const Assignments &assignments_)
      : assignments (assignments_)
    {}


    // Return true iff LEFT has more unassigned neighbors than those
    // of RIGHT with ties being broken by lexicographical order of the
    // variables' names.
    bool 
    operator() (const Variable *left, const Variable *right) const
    {
      assert (left != NULL);
      assert (right != NULL);

      const NeighborCount left_unass_count (unassigned_neighbors_count (left));
      const NeighborCount right_unass_count 
	(unassigned_neighbors_count (right));


      // Note here that larger degrees come first in DEG.
      return (left_unass_count > right_unass_count
	      || (!(right_unass_count > left_unass_count)
		  && name (left) < name (right)));
    }
  };


  // A binary predicate for sorting variables by domain-degree ratio
  // (DDR).
  class DdrPredicate : public LdPredicate, public DegPredicate
  {
  private:

    double
    ratio (const Variable *v) const
    {
      assert (v != NULL);

      NeighborCount unass_count (unassigned_neighbors_count (v));

      // To avoid dividing by zero, if the number of unassigned
      // neighbors is zero, set it to one.
      if (unass_count == 0)
	unass_count = 1;

      return (double (domain_size (v)) / unass_count);
    }

  public:
    // Construct a DDR predicate.
    DdrPredicate (const CurrentDomains &current_domains_,
		  const Assignments &assignments_)
      : LdPredicate (current_domains_),
	DegPredicate (assignments_)
    {
      assert (assignments.size () <= current_domains.size ());
    }


    // Return true iff LEFT has a smaller domain-degree ratio than
    // that of RIGHT with ties being broken by lexicographical order
    // of the variables' names.
    bool 
    operator() (const Variable *left, const Variable *right) const
    {
      assert (left != NULL);
      assert (right != NULL);

      const double left_ratio (ratio (left));
      const double right_ratio (ratio (right));

      return (left_ratio < right_ratio
	      || (!(right_ratio < left_ratio)
		  && name (left) < name (right)));
    }
  };


  // Order the current path in the range [first, last) by this
  // solver's ordering heuristic.
  void 
  SystematicSearchSolver::
  order_current_path (SearchPath::iterator first,
		      SearchPath::iterator last)
  {
    assert (first > current_path.begin ());
    assert (last == current_path.end ()); // For now.
    assert (current_domains.size () == num_vars () + 1);
    assert (current_domains.find (dummy) != current_domains.end ());
    assert (current_domains[dummy].empty ());
    assert (current_path.size () == num_vars () + 1);
    assert (current_path.front () == dummy);
    assert (assignments.size () <= num_vars () + 1);

    switch (var_order)
      {
      case NO_VAR_ORDER:
	break;
      case LEAST_DOMAIN:
	std::sort (first, last, LdPredicate (current_domains));
	break;
      case DEGREE:
	std::sort (first, last, DegPredicate (assignments));
	break;
      case DOMAIN_DEGREE_RATIO:
	std::sort (first, last, DdrPredicate (current_domains, assignments));
	break;
      default:
	// Unreachable.
	assert (false);
      }

    assert (first > current_path.begin ());
    assert (last == current_path.end ()); // For now.
    assert (current_domains.size () == num_vars () + 1);
    assert (current_domains.find (dummy) != current_domains.end ());
    assert (current_domains[dummy].empty ());
    assert (current_path.size () == num_vars () + 1);
    assert (current_path.front () == dummy);
    assert (assignments.size () <= num_vars () + 1);
  }


  // Return a const reference to the current domain for variable VAR.
  const CurrentDomains::mapped_type &
  SystematicSearchSolver::
  get_current_domain_at (const CurrentDomains::key_type &var) const
  {
    assert (var != NULL);
    assert (current_domains.find (var) != current_domains.end ());
    assert (current_domains.size () == num_vars () + 1);
    assert (current_domains.find (dummy) != current_domains.end ());
    assert (current_domains.find (dummy)->second.empty ());
    assert (current_path.size () == num_vars () + 1);
    assert (current_path.front () == dummy);
    assert (!assignments.empty ());
    assert (assignments.size () <= num_vars () + 1);

    return current_domains.find (var)->second;
  }

  
  // Remove value VAL from the current domain of variable VAR.  Watch
  // out for invalidating iterators when iterating over a domain.
  void
  SystematicSearchSolver::
  remove_from_current_domain_at (const CurrentDomains::key_type &var, 
				 const CurrentDomains::mapped_type::
				 value_type &val)
  {
    assert (var != NULL);
    assert (current_domains.find (var) != current_domains.end ());
    assert (var == dummy
	    || (current_domains.find (var)->second.find (val)
		!= current_domains.find (var)->second.end ()));
    assert (current_domains.size () == num_vars () + 1);
    assert (current_domains.find (dummy) != current_domains.end ());
    assert (current_domains[dummy].empty ());
    assert (current_path.size () == num_vars () + 1);
    assert (current_path.front () == dummy);
    assert (!assignments.empty ());
    assert (assignments.size () <= num_vars () + 1);

    current_domains[var].erase (val);

    assert (var != NULL);
    assert (current_domains.find (var) != current_domains.end ());
    assert (current_domains[var].find (val) == current_domains[var].end ());
    assert (current_domains.size () == num_vars () + 1);
    assert (current_domains.find (dummy) != current_domains.end ());
    assert (current_domains[dummy].empty ());
    assert (current_path.size () == num_vars () + 1);
    assert (current_path.front () == dummy);
    assert (!assignments.empty ());
    assert (assignments.size () <= num_vars () + 1);
  }


  // Set the current domain of varoable VAR to domain DOM.
  void
  SystematicSearchSolver::
  set_current_domain_at (const CurrentDomains::key_type &var,
			 const CurrentDomains::mapped_type &dom)
  {
    assert (var != NULL);
    assert (current_domains.find (var) != current_domains.end ());
    assert (current_domains.size () == num_vars () + 1);
    assert (current_domains.find (dummy) != current_domains.end ());
    assert (current_domains[dummy].empty ());
    assert (current_path.size () == num_vars () + 1);
    assert (current_path.front () == dummy);
    assert (!assignments.empty ());
    assert (assignments.size () <= num_vars () + 1);

    current_domains[var] = dom;

    assert (var != NULL);
    assert (current_domains.find (var) != current_domains.end ());
    assert (current_domains.size () == num_vars () + 1);
    assert (current_domains.find (dummy) != current_domains.end ());
    assert (current_domains[dummy].empty ());
    assert (current_path.size () == num_vars () + 1);
    assert (current_path.front () == dummy);
    assert (!assignments.empty ());
    assert (assignments.size () <= num_vars () + 1);
  }


  // Reset the current domain of variable VAR to its initial domain.
  void
  SystematicSearchSolver::
  reset_current_domain_at (const CurrentDomains::key_type &var)
  {
    assert (var != NULL);
    assert (current_domains.find (var) != current_domains.end ());
    assert (current_domains.size () == num_vars () + 1);
    assert (current_domains.find (dummy) != current_domains.end ());
    assert (current_domains[dummy].empty ());
    assert (current_path.size () == num_vars () + 1);
    assert (current_path.front () == dummy);
    assert (!assignments.empty ());
    assert (assignments.size () <= num_vars () + 1);

    current_domains[var] = var->get_initial_domain ();

    assert (var != NULL);
    assert (current_domains.find (var) != current_domains.end ());
    assert (current_domains.size () == num_vars () + 1);
    assert (current_domains.find (dummy) != current_domains.end ());
    assert (current_domains[dummy].empty ());
    assert (current_path.size () == num_vars () + 1);
    assert (current_path.front () == dummy);
    assert (!assignments.empty ());
    assert (assignments.size () <= num_vars () + 1);
  }


  const SearchPath &
  SystematicSearchSolver::
  get_current_path (void) const
  {
    assert (current_domains.size () == num_vars () + 1);
    assert (current_path.size () == num_vars () + 1);
    assert (current_path.front () == dummy);
    assert (assignments.size () <= num_vars () + 1);
    
    return current_path;
  }


  // TODO: get rid of this.
  SearchPath &
  SystematicSearchSolver::
  get_current_path (void)
  {
    assert (current_domains.size () == num_vars () + 1);
    assert (current_path.size () == num_vars () + 1);
    assert (current_path.front () == dummy);
    assert (assignments.size () <= num_vars () + 1);
    
    return current_path;
  }


  // Assign the value VAL to the variable at I in CURRENT_PATH.  The
  // variable at I may already have an assigned value.  This method
  // must use a CURRENT_PATH iterator instead of a variable since this
  // method may reorder CURRENT_PATH.
  void
  SystematicSearchSolver::
  assign (SearchPath::iterator i, const Assignments::mapped_type &val)
  {
    assert (i > current_path.begin ());
    assert (i < current_path.end ());
    assert (*i != NULL);
    assert (current_domains.size () == num_vars () + 1);
    assert (current_domains.find (dummy) != current_domains.end ());
    assert (current_domains[dummy].empty ());
    assert (current_path.size () == num_vars () + 1);
    assert (current_path.front () == dummy);
    assert (!assignments.empty ());
    assert (assignments.size () <= num_vars () + 1);

    // This is true iff we are updating an assignment; that is, iff it
    // is currently assigned.
    bool updating_assignment (assignments.find (*i) != assignments.end ());

    // Make the assignment.
    assignments[*i] = val;

    // If we are using dynamic variable ordering and this is not an
    // update (which occurs when iterating through a variable's
    // domain), then reorder CURRENT_PATH from I + 1 to the end.  This
    // sets up the next variable in the search path.
    if (dynamic_var_order && !updating_assignment)
      order_current_path (i + 1, current_path.end ());

    assert (i > current_path.begin ());
    assert (i < current_path.end ());
    assert (*i != NULL);
    assert (assignments.find (*i) != assignments.end ());
    assert (current_domains.size () == num_vars () + 1);
    assert (current_domains.find (dummy) != current_domains.end ());
    assert (current_domains[dummy].empty ());
    assert (current_path.size () == num_vars () + 1);
    assert (current_path.front () == dummy);
    assert (!assignments.empty ());
    assert (assignments.size () <= num_vars () + 1);
  }


  // Remove the current assignment from each variable in the range
  // [first, last) of the current path.
  void
  SystematicSearchSolver::
  unassign (SearchPath::const_iterator first,
	    SearchPath::const_iterator last)
  {
    assert (first > current_path.begin ());
    assert (last >= first);
    assert (current_domains.size () == num_vars () + 1);
    assert (current_domains.find (dummy) != current_domains.end ());
    assert (current_domains[dummy].empty ());
    assert (current_path.size () == num_vars () + 1);
    assert (current_path.front () == dummy);
    assert (!assignments.empty ());
    assert (assignments.size () <= num_vars () + 1);

    for (SearchPath::const_iterator i (first); i != last; ++i)
      {
	// If the current domain for *I is empty, then it wasn't
	// assigned a value, so move on.
	if (get_current_domain_at (*i).empty ())
	  {
	    assert (assignments.find (*i) == assignments.end ());
	    continue;
	  }

	assert (i > current_path.begin ());
	assert (i < current_path.end ());
	assert (*i != NULL);
	assert (assignments.find (*i) != assignments.end ());
	
	assignments.erase (*i);
      }

    assert (first > current_path.begin ());
    assert (last >= first);
    assert (current_domains.size () == num_vars () + 1);
    assert (current_domains.find (dummy) != current_domains.end ());
    assert (current_domains[dummy].empty ());
    assert (current_path.size () == num_vars () + 1);
    assert (current_path.front () == dummy);
    assert (!assignments.empty ());
    assert (assignments.size () <= num_vars () + 1);
  }


  // Return the value of the variable V.
  const Assignments::mapped_type &
  SystematicSearchSolver::
  get_value (const Assignments::key_type &var) const
  {
    assert (var != NULL);
    assert (assignments.find (var) != assignments.end ());
    assert (current_domains.size () == num_vars () + 1);
    assert (current_domains.find (dummy) != current_domains.end ());
    assert (current_domains.find (dummy)->second.empty ());
    assert (current_path.size () == num_vars () + 1);
    assert (current_path.front () == dummy);
    assert (!assignments.empty ());
    assert (assignments.size () <= num_vars () + 1);

    return assignments.find (var)->second;

    assert (var != NULL);
    assert (assignments.find (var) != assignments.end ());
    assert (current_domains.size () == num_vars () + 1);
    assert (current_domains.find (dummy) != current_domains.end ());
    assert (current_domains.find (dummy)->second.empty ());
    assert (current_path.size () == num_vars () + 1);
    assert (current_path.front () == dummy);
    assert (!assignments.empty ());
    assert (assignments.size () <= num_vars () + 1);
  }
  

  // Return the name of this solvers ordering heuristic.
  std::string 
  SystematicSearchSolver::
  ordering_string (void) const
  {
    std::string ordering;

    switch (var_order)
      {
      case NO_VAR_ORDER:
	ordering = "NONE";
	break;
      case LEAST_DOMAIN:
	ordering = "LD";
	break;
      case DEGREE:
	ordering = "DEG";
	break;
      case DOMAIN_DEGREE_RATIO:
	ordering = "DDR";
	break;
      default:
	// Unreachable.
	assert (false);
      }

    if (dynamic_var_order)
      ordering += "-d";

    return ordering;
  }


  // Status of search in bcssp.
  enum Status
  {
    UNKNOWN,
    SOLUTION,
    IMPOSSIBLE
  };
  
  
  // This function is based on Patrick Prosser, Hybrid Algorithms for
  // the Constraint Satisfaction Problem.
  void
  bcssp (SystematicSearchSolver &solver)
  {
    assert (solver.cc == 0);
    assert (solver.nv == 0);
    assert (solver.bt == 0);
    assert (solver.num_solutions == 0);
    assert (solver.current_domains.size () == solver.num_vars () + 1);
    assert (solver.current_domains.find (solver.dummy)
	    != solver.current_domains.end ());
    assert (solver.current_path.size () == solver.num_vars () + 1);
    assert (solver.current_path.front () == solver.dummy);
    assert (solver.assignments.size () == 1);

    // consistent <-- true;
    bool consistent (true);

    // status <-- "unknown";
    Status status (UNKNOWN);

    // i <-- 1;
    SearchPath::iterator i (solver.current_path.begin () + 1);
    
    // WHILE status = "unknown"
    while (status == UNKNOWN)
      {
	assert (i >= solver.current_path.begin ());
	assert (i <= solver.current_path.end ());

	// IF consistent
	if (consistent)
	  {
	    // i <-- label (i, consistent);
	    i = solver.label (i, consistent);
	  }
	else
	  {
	    // Keep a record of the currently instantiated variable.
	    SearchPath::const_iterator j (i);

	    // i <-- unlabel (i, consistent);
	    i = solver.unlabel (i, consistent);

	    // Unassign all variables uninstantiated by the unlabel
	    // method.  Like STL algorithms, the second argument is an
	    // iterator to one past the last element.
	    solver.unassign (i + 1, j + 1);
	  }

	// IF i > n
	if (i == solver.current_path.end ())
	  {
	    // status <-- "solution";
	    status = SOLUTION;

// 	    std::cerr << "Solution: " << solver.assignments << std::endl;

	    // Increment the number of solutions found.
	    ++solver.num_solutions;

	    // If we want all solutions, then we need to keep going.
	    // This is handled here.
	    if (solver.find_all)
	      {
		// This is a specialized unlabel method, not used by
		// Prosser to unlabel a solution.  The
		// unlabel_solution methods handle any special needs
		// of finding more solutions.
		i = solver.unlabel_solution (consistent);

		// As with a normal unlabel (see above), we need to
		// unassign the variables uninstantiated by the
		// unlabel_solutions method.
		solver.unassign (i + 1, solver.current_path.end ());

		// Reset the status to "unknown" since there may be
		// more solutions.
		status = UNKNOWN;
	      }
	  }
	// ELSE IF i = 0
	else if (i == solver.current_path.begin ())
	  {
	    // status <-- "impossible"
	    status = IMPOSSIBLE;
	  }
      }
    assert (!solver.find_all || i == solver.current_path.begin ());

    assert (solver.find_all || solver.num_solutions <= 1);
    assert (solver.current_domains.size () == solver.num_vars () + 1);
    assert (solver.current_domains.find (solver.dummy)
	    != solver.current_domains.end ());
    assert (solver.current_path.size () == solver.num_vars () + 1);
    assert (solver.current_path.front () == solver.dummy);
  }


  // This method is based on Prosser's bt-label.
  SearchPath::iterator
  BtSolver::
  label (SearchPath::iterator i, bool &consistent)
  {
    assert (i != get_current_path ().end ());    
    assert (*i != NULL);

    // consistent <-- false;
    consistent = false;
    
    // FOR v[i] <-- EACH ELEMENT OF current-domain[i] WHILE not
    // consistent
    for (Domain::iterator val = get_current_domain_at (*i).begin (); 
	 !consistent && val != get_current_domain_at (*i).end (); )
      {
	// Instantiate the current variable with the next value in its
	// domain.
	assign (i, *val);

	// Increment the number of nodes visited since a new
	// assignment was made.
	++nv;

	// consistent <-- true;
	consistent = true;

	// FOR h <-- 1 TO i - 1 WHILE consistent
	for (SearchPath::iterator h (get_current_path ().begin () + 1);
	     h <= i - 1 && consistent; ++h)
	  {
	    // consistent <-- check (i, h);
	    const VvpSet vvps (*i, get_value (*i), *h, get_value (*h));
	    consistent = problem->check(vvps, NULL);
	    
	    // Increment the total number of constraint checks.
	    ++cc;
	  }

	// IF not consistent
	if (!consistent)
	  // current-domain[i] <-- remove (v[i], current-domain[i])
	  remove_from_current_domain_at (*i, *val++);
	else
	  ++val;
      }

    // IF consistent THEN return (i + 1) ELSE return (i)
    if (consistent)
      return i + 1;
    else
      return i;
  }


  // This method is based on Prosser's bt-unlabel.
  SearchPath::iterator
  BtSolver::
  unlabel (SearchPath::iterator i, bool &consistent)
  {
    assert (i != get_current_path ().end ());    
    assert (*i != NULL);

    // Increment the number of backtracks.
    ++bt;

    // h <-- i - 1;
    SearchPath::iterator h (i - 1);

    // current-domain[i] <-- domain[i];
    reset_current_domain_at (*i);

    // current-domain[h] <-- remove (v[h], current-domain[h]);
    remove_from_current_domain_at (*h, get_value (*h));

    // consistent <-- current-domain[h] /= nil;
    consistent = !get_current_domain_at (*h).empty ();

    // return (h)
    return h;
  }


  // A special case of bt-unlabel used after a solution is found.
  SearchPath::iterator
  BtSolver::
  unlabel_solution (bool &consistent)
  {
    // Increment the number of backtracks.
    ++bt;

    // H is the last variable of CURRENT_PATH.
    SearchPath::iterator h (get_current_path ().end () - 1);

    // current-domain[h] <-- remove (v[h], current-domain[h]);
    remove_from_current_domain_at (*h, get_value (*h));

    // consistent <-- current-domain[h] /= nil;
    consistent = !get_current_domain_at (*h).empty ();

    // return (h)
    return h;
  }


  // Solve the problem with this backtracking solver.
  inline void
  BtSolver::
  solve (void)
  {
    // Get the start click.
    solve_begin = std::clock ();

    // Find solution(s).
    bcssp (*this);

    // Get the stop click.
    solve_end = std::clock ();

    // Output stats.
    std::cout << std::left 
	      << std::setw (8)
	      << ((ordering_string () != "NONE")
		  ? algorithm_string () : algorithm_string () + ':') 
	      << std::setw (6)
	      << ((ordering_string () != "NONE")
		  ? ordering_string () + ':' : "")
	      << " CC: "  << std::setw (7) << cc 
	      << " NV: "  << std::setw (7) << nv
	      << " BT: "  << std::setw (7) << bt
	      << " CPU: " << std::setw (6) << std::setprecision (6)
	      << ((double) (solve_end - solve_begin)) / CLOCKS_PER_SEC
	      << " Sols: " << std::setw (3) << num_solutions << std::endl;
  }


  // The name of this solver's algorithm.
  std::string
  BtSolver::
  algorithm_string (void) const
  {
    return "BT";
  }


  // Initialize REDUCTIONS.
  void
  FcSolver::
  init_reductions (void)
  {
    assert (solve_begin == 0);
    assert (solve_end == 0);
    assert (cc == 0);
    assert (nv == 0);
    assert (bt == 0);
    assert (num_solutions == 0);
    assert (reductions.empty ());

    // Initialize each element of REDUCTIONS to empty.
    for (VariableVector::const_iterator i (problem->get_variables ().begin ()); 
	 i != problem->get_variables ().end (); ++i)
      {
	assert (*i != NULL);
	assert (reductions.find (*i) == reductions.end ());

	reductions.insert (std::make_pair (*i, Reductions::mapped_type ()));

	assert (reductions.find (*i) != reductions.end ());
	assert (reductions.find (*i)->second.empty ());
      }

    assert (solve_begin == 0);
    assert (solve_end == 0);
    assert (cc == 0);
    assert (nv == 0);
    assert (bt == 0);
    assert (num_solutions == 0);
    assert (reductions.size () == num_vars ());
  }


  // Initialize PAST_FCS.
  void
  FcSolver::
  init_past_fcs (void)
  {
    assert (solve_begin == 0);
    assert (solve_end == 0);
    assert (cc == 0);
    assert (nv == 0);
    assert (bt == 0);
    assert (num_solutions == 0);
    assert (past_fcs.empty ());

    // Initialize each element of PAST_FC to DUMMY, the first element
    // of CURRENT_PATH.
    for (SearchPath::const_iterator i = get_current_path ().begin (); 
	 i != get_current_path ().end (); ++i)
      {
	assert (past_fcs.find (*i) == past_fcs.end ());

	// We use a CURRENT_PATH iterator since PAST_FCS maps
	// variables to a deque of CURRENT_PATH iterators.
	const PastFcs::mapped_type data (1, get_current_path ().begin ());
	past_fcs.insert (std::make_pair (*i, data));

	assert (past_fcs.find (*i) != past_fcs.end ());
	assert (past_fcs.find (*i)->second.size () == 1);
	assert (past_fcs.find (*i)->second.front () 
		== get_current_path ().begin ());
      }

    assert (solve_begin == 0);
    assert (solve_end == 0);
    assert (cc == 0);
    assert (nv == 0);
    assert (bt == 0);
    assert (num_solutions == 0);
    assert (past_fcs.size () == num_vars () + 1);
  }


  // Initialize FUTURE_FCS.
  void
  FcSolver::
  init_future_fcs (void)
  {
    assert (solve_begin == 0);
    assert (solve_end == 0);
    assert (cc == 0);
    assert (nv == 0);
    assert (bt == 0);
    assert (num_solutions == 0);
    assert (future_fcs.empty ());

    // Initialize each element of FUTURE_FCS to empty.
    for (SearchPath::const_iterator i = get_current_path ().begin (); 
	 i != get_current_path ().end (); ++i)
      {
	assert (future_fcs.find (*i) == future_fcs.end ());

	future_fcs.insert (std::make_pair (*i, FutureFcs::mapped_type ()));

	assert (future_fcs.find (*i) != future_fcs.end ());
	assert (future_fcs.find (*i)->second.empty ());
      }

    assert (solve_begin == 0);
    assert (solve_end == 0);
    assert (cc == 0);
    assert (nv == 0);
    assert (bt == 0);
    assert (num_solutions == 0);
    assert (future_fcs.size () == num_vars () + 1);
  }
  

  // Push REDUCTION onto the stack (deque) of reductions for variable
  // VAR.
  void
  FcSolver::
  push_onto_reductions_at (const Reductions::key_type &var,
			   const Reductions::mapped_type::value_type &reduction)
  {
    assert (reductions.find (var) != reductions.end ());
    assert (reductions.size () == num_vars ());
    assert (past_fcs.size () == num_vars () + 1);
    assert (future_fcs.size () == num_vars () + 1);

    reductions[var].push_front (reduction);

    assert (reductions.find (var) != reductions.end ());
    assert (reductions[var].front () == reduction);
    assert (reductions.size () == num_vars ());
    assert (past_fcs.size () == num_vars () + 1);
    assert (future_fcs.size () == num_vars () + 1);
  }


  // Get the stack (deque) of reductions for variable VAR.
  const Reductions::mapped_type &
  FcSolver::
  get_reductions_at (const Reductions::key_type &var) const
  {
    assert (reductions.find (var) != reductions.end ());
    assert (reductions.size () == num_vars ());
    assert (past_fcs.size () == num_vars () + 1);
    assert (future_fcs.size () == num_vars () + 1);

    return reductions.find (var)->second;
  }


  // Treating each reduction of REDUCTIONS as a stack, return a const
  // reference to the top reduction for variable VAR.
  const Reductions::mapped_type::value_type &
  FcSolver::
  top_reduction_at (const Reductions::key_type &var) const
  {
    assert (reductions.find (var) != reductions.end ());
    assert (!reductions.find (var)->second.empty ());
    assert (reductions.size () == num_vars ());
    assert (past_fcs.size () == num_vars () + 1);
    assert (future_fcs.size () == num_vars () + 1);

    // The front of the deque is considered the top.
    return reductions.find (var)->second.front ();
  }


  // Treating each reduction of REDUCTIONS as a stack, pop the top
  // reduction for variable VAR.  Use top_reduction_at to obtain the
  // value of the top variable.
  void 
  FcSolver::
  pop_reduction_at (const Reductions::key_type &var)
  {
    assert (reductions.find (var) != reductions.end ());
    assert (!reductions[var].empty ());
    assert (reductions.size () == num_vars ());
    assert (past_fcs.size () == num_vars () + 1);
    assert (future_fcs.size () == num_vars () + 1);

    // The front of the deque is considered the top.
    reductions[var].pop_front ();

    assert (reductions.find (var) != reductions.end ());
    assert (reductions.size () == num_vars ());
    assert (past_fcs.size () == num_vars () + 1);
    assert (future_fcs.size () == num_vars () + 1);
  }


  void
  FcSolver::
  push_onto_future_fcs_at (const FutureFcs::key_type &var,
			   const FutureFcs::mapped_type::value_type &future)
  {
    assert (future_fcs.find (var) != future_fcs.end ());
    assert (reductions.size () == num_vars ());
    assert (past_fcs.size () == num_vars () + 1);
    assert (future_fcs.size () == num_vars () + 1);

    future_fcs[var].push_front (future);

    assert (future_fcs.find (var) != future_fcs.end ());
    assert (future_fcs[var].front () == future);
    assert (reductions.size () == num_vars ());
    assert (past_fcs.size () == num_vars () + 1);
    assert (future_fcs.size () == num_vars () + 1);
  }

  
  const FutureFcs::mapped_type &
  FcSolver::
  get_future_fcs_at (const FutureFcs::key_type &var) const
  {
    assert (future_fcs.find (var) != future_fcs.end ());
    assert (reductions.size () == num_vars ());
    assert (past_fcs.size () == num_vars () + 1);
    assert (future_fcs.size () == num_vars () + 1);

    return future_fcs.find (var)->second;
  }
  

  void
  FcSolver::
  clear_future_fcs_at (const FutureFcs::key_type &var)
  {
    assert (future_fcs.find (var) != future_fcs.end ());
    assert (reductions.size () == num_vars ());
    assert (past_fcs.size () == num_vars () + 1);
    assert (future_fcs.size () == num_vars () + 1);

    future_fcs[var].clear ();

    assert (future_fcs.find (var) != future_fcs.end ());
    assert (future_fcs.find (var)->second.empty ());
    assert (reductions.size () == num_vars ());
    assert (past_fcs.size () == num_vars () + 1);
    assert (future_fcs.size () == num_vars () + 1);
  }

  
  void
  FcSolver::
  push_onto_past_fcs_at (const PastFcs::key_type &var,
			  const PastFcs::mapped_type::value_type &past)
  {
    assert (past_fcs.find (var) != past_fcs.end ());
    assert (!past_fcs.find (var)->second.empty ());
    assert (past_fcs.find (var)->second.back ()
	    == get_current_path ().begin ());
    assert (reductions.size () == num_vars ());
    assert (past_fcs.size () == num_vars () + 1);
    assert (future_fcs.size () == num_vars () + 1);

    past_fcs[var].push_front (past);

    assert (past_fcs.find (var) != past_fcs.end ());
    assert (!past_fcs.find (var)->second.empty ());
    assert (past_fcs.find (var)->second.back ()
	    == get_current_path ().begin ());
    assert (past_fcs[var].front () == past);
    assert (reductions.size () == num_vars ());
    assert (past_fcs.size () == num_vars () + 1);
    assert (future_fcs.size () == num_vars () + 1);
  }

  
  const PastFcs::mapped_type &
  FcSolver::
  get_past_fcs_at (const PastFcs::key_type &var)
  {
    assert (past_fcs.find (var) != past_fcs.end ());
    assert (!past_fcs.find (var)->second.empty ());
    assert (past_fcs.find (var)->second.back ()
	    == get_current_path ().begin ());
    assert (reductions.size () == num_vars ());
    assert (past_fcs.size () == num_vars () + 1);
    assert (future_fcs.size () == num_vars () + 1);

    return past_fcs.find (var)->second;
  }
  

  void 
  FcSolver::
  pop_past_fcs_at (const PastFcs::key_type &var)
  {
    assert (past_fcs.find (var) != past_fcs.end ());
    assert (!past_fcs.find (var)->second.empty ());
    assert (past_fcs.find (var)->second.back ()
	    == get_current_path ().begin ());
    assert (!past_fcs[var].empty ());
    assert (reductions.size () == num_vars ());
    assert (past_fcs.size () == num_vars () + 1);
    assert (future_fcs.size () == num_vars () + 1);

    past_fcs[var].pop_front ();

    assert (past_fcs.find (var) != past_fcs.end ());
    assert (!past_fcs.find (var)->second.empty ());
    assert (past_fcs.find (var)->second.back ()
	    == get_current_path ().begin ());
    assert (reductions.size () == num_vars ());
    assert (past_fcs.size () == num_vars () + 1);
    assert (future_fcs.size () == num_vars () + 1);
  }

  
  bool
  FcSolver::
  check_forward (SearchPath::iterator i, SearchPath::iterator j)
  {
    assert (*i != NULL);
    assert (*j != NULL);
    assert (i != get_current_path ().end ());
    assert (j != get_current_path ().end ());
    assert (past_fcs.find (*i) != past_fcs.end ());
    assert (past_fcs.find (*j) != past_fcs.end ());
    assert (!past_fcs.find (*i)->second.empty ());
    assert (!past_fcs.find (*j)->second.empty ());
    assert (past_fcs.find (*i)->second.back () == get_current_path ().begin ());
    assert (past_fcs.find (*j)->second.back () == get_current_path ().begin ());

    // reduction <-- nil.
    Domain reduction;

    // FOR v[j] <-- EACH ELEMENT OF current-domain[j]:
    for (Domain::iterator val (get_current_domain_at (*j).begin ());
	 val != get_current_domain_at (*j).end (); ++val)
      {
	// Instantiate the future variable J with the next value in
	// its domain.
	assign (j, *val);

	// Increment the number of nodes visited since a new
	// assignment was made.
	++nv;

	// consistent <-- check (i, j).
	const VvpSet vvps (*i, get_value (*i), *j, get_value (*j));;
	bool consistent (problem->check (vvps, NULL));

	// Increment the total number of constraint checks.
	++cc;

	// IF not consistent:
	if (!consistent)
	  {
	    // push (v[j], reduction)
	    reduction.insert (*val); // TODO: is this good?
	  }
      }

    // If reduction /= nil:
    if (!reduction.empty ())
      {
	// current-domain[j] <-- set-difference (current-domain[j], reduction).
	Domain difference;
	std::insert_iterator<Domain> differ_inserter (difference, 
						      difference.end ());
	std::set_difference (get_current_domain_at (*j).begin (),
			     get_current_domain_at (*j).end (),
			     reduction.begin (), reduction.end (),
			     differ_inserter);
	set_current_domain_at (*j, difference);

	// push (reduction, reductions[j])
	push_onto_reductions_at (*j, reduction);
	
	// push (j, future-fc[i])
	push_onto_future_fcs_at (*i, *j);

	// push (i, past-fc[j])
	push_onto_past_fcs_at (*j, i);
      }

    // return (current-domain[j] /= nil).
    return !get_current_domain_at (*j).empty ();
  }


  void
  FcSolver::
  undo_reductions (SearchPath::iterator i)
  {
    assert (i != get_current_path ().end ());
    assert (*i != NULL);

    // FOR j <-- EACH ELEMENT OF future-fc[i]:
    for (FutureFcsDeque::const_iterator j (get_future_fcs_at (*i).begin ());
	 j != get_future_fcs_at (*i).end (); ++j)
      {
	// reduction <-- pop (reductions[j]);
	const Domain &reduction (top_reduction_at (*j));
	// This is actually popped below.
	assert (!reduction.empty ()); // Maybe not.

	// current-domain[j] <-- union (current-domain[j], reduction).
	Domain unionx;
	std::insert_iterator<Domain> union_inserter (unionx, unionx.end ());
	// Domains are sets, so they are already sorted.
	std::set_union (get_current_domain_at (*j).begin (),
			get_current_domain_at (*j).end (),
			reduction.begin (),
			reduction.end (), union_inserter);
	set_current_domain_at (*j, unionx);

	// Now pop <-- reductions[j];
	pop_reduction_at (*j);

	// pop (past-fc[j])
	pop_past_fcs_at (*j);
      }

    // future-fc[i] <-- nil
    clear_future_fcs_at (*i);
  }


  void
  FcSolver::
  update_current_domain (SearchPath::iterator i)
  {
    assert (i != get_current_path ().end ());
    assert (*i != NULL);

    // current-domain[i] = domain[i];
    reset_current_domain_at (*i);

    // FOR reduction <-- EACH ELEMENT OF reductions[i]
    for (ReductionsDeque::const_iterator reduction
	   (get_reductions_at (*i).begin ());
	 reduction != get_reductions_at (*i).end (); ++reduction)
      {
	// current-domain[i] <-- set-difference (current-domain[i], reduction)
	Domain difference;
	// Domains are sets, so they are already sorted.
	std::insert_iterator<Domain> differ_inserter (difference, 
						      difference.end ());
	std::set_difference (get_current_domain_at (*i).begin (),
			     get_current_domain_at (*i).end (),
			     reduction->begin (),
			     reduction->end (), differ_inserter);
	set_current_domain_at (*i, difference);
      }
  }


  SearchPath::iterator
  FcSolver::
  label (SearchPath::iterator i, bool &consistent)
  {
    assert (i != get_current_path ().end ());
    assert (*i != NULL);

    // consistent <-- false.
    consistent = false;

    // FOR v[i] <-- EACH ELEMENT OF current-domain[i] WHILE not consistent:
    for (Domain::iterator val (get_current_domain_at (*i).begin ()); 
	 !consistent && val != get_current_domain_at (*i).end (); )
      {
	// Instantiate the current variable with the next value in its
	// domain.
	assign (i, *val);

	// Increment the number of nodes visited since a new
	// assignment was made.
	++nv;

	// consistent <-- true.
	consistent = true;

	// FOR j <-- i + 1 TO n WHILE consistent:
	for (SearchPath::iterator j (i + 1);
	     j != get_current_path ().end () && consistent; ++j)
	  // consistent <-- check-forward (i, j).
	  consistent = check_forward (i, j);

	// IF not consistent:
	if (!consistent)
	  {
	    // current-domain[i] <-- remove (v[i], current-domain[i]).
	    remove_from_current_domain_at (*i, *val++);

	    // undo-reductions (i)
	    undo_reductions (i);
	  }
	else
	  {
	    ++val;
	  }
      }

    // IF consistent THEN return (i + 1) ELSE return (i).
    if (consistent)
      return i + 1;
    else
      return i;
  }


  SearchPath::iterator
  FcSolver::
  unlabel (SearchPath::iterator i, bool &consistent)
  {
    assert (i != get_current_path ().end ());
    assert (*i != NULL);

    // Increment the number of backtracks.
    ++bt;
    
    // h <-- i - 1.
    SearchPath::iterator h (i - 1);

    undo_reductions (h);
    update_current_domain (i);

    // current-domain[h] <-- remove (v[h], current-domain[h]).
    remove_from_current_domain_at (*h, get_value (*h));

    // consistent <-- current-domain[h] /= nil.
    consistent = !get_current_domain_at (*h).empty ();

    // return (h).
    return h;
  }


  SearchPath::iterator
  FcSolver::
  unlabel_solution (bool &consistent)
  {
    // Increment the number of backtracks.
    ++bt;

    SearchPath::iterator h (get_current_path ().end () - 1);
    undo_reductions (h);
    remove_from_current_domain_at (*h, get_value (*h));
    consistent = !get_current_domain_at (*h).empty ();
    return h;
  }


  std::string 
  FcSolver::
  algorithm_string (void) const
  {
    return "FC";
  }


  // Initialize each conflict set to {DUMMY}.
  void
  CbjSolver::
  init_conf_sets (void)
  {
    assert (conf_sets.empty ());

    for (SearchPath::const_iterator i = get_current_path ().begin (); 
	 i != get_current_path ().end (); ++i)
      {
	assert (*i != NULL);
	assert (conf_sets.find (*i) == conf_sets.end ());
	assert (get_current_path ().front () == dummy);

	SearchPathIterSet conf_set;
	conf_set.insert (get_current_path ().begin ());
	conf_sets.insert (std::make_pair (*i, conf_set));

	assert (conf_sets.find (*i) != conf_sets.end ());
      }
      
    assert (conf_sets.size () == num_vars () + 1);
  }


   void
   CbjSolver::
   set_conf_set_at (const SearchPath::value_type &var,
 		   const ConfSets::mapped_type &conf_set)
   {
     assert (conf_sets.find (var) != conf_sets.end ());
     assert (conf_sets.size () == num_vars () + 1);
 
     conf_sets[var] = conf_set;
 
     assert (conf_sets.find (var) != conf_sets.end ());
     assert (conf_sets.size () == num_vars () + 1);
   }
 
 
   void
   CbjSolver::
   insert_into_conf_set_at (const SearchPath::value_type &var,
 			   SearchPath::iterator i)
   {
     assert (conf_sets.find (var) != conf_sets.end ());
     assert (conf_sets.size () == num_vars () + 1);
 
     conf_sets[var].insert (conf_sets[var].end (), i);
 
     assert (conf_sets.find (var) != conf_sets.end ());    
     assert (conf_sets.find (var)->second.find (i)
 	    != conf_sets.find (var)->second.end ());
     assert (conf_sets.size () == num_vars () + 1);
   }
 
 
   // Reset CONF_SET for variable VAR to {DUMMY}.
   void
   CbjSolver::
   reset_conf_set_at (const SearchPath::value_type &var)
   {
     assert (conf_sets.find (var) != conf_sets.end ());
     assert (!conf_sets.find (var)->second.empty ());
     assert (conf_sets.size () == num_vars () + 1);
 
     conf_sets[var].clear ();
     conf_sets[var].insert (conf_sets[var].begin (),
 			   get_current_path ().begin ());
 
     assert (conf_sets.find (var) != conf_sets.end ());
     assert (!conf_sets.find (var)->second.empty ());
     assert (conf_sets.size () == num_vars () + 1);
   }
 
 
   const CbjSolver::ConfSets::mapped_type &
   CbjSolver::
   conf_set_at (const SearchPath::value_type &var) const
   {
     assert (conf_sets.find (var) != conf_sets.end ());
     assert (!conf_sets.find (var)->second.empty ());
     assert (conf_sets.size () == num_vars () + 1);
 
     return conf_sets.find (var)->second;
   }
 
 
  // This method is based on Prosser's cbj-label.
  SearchPath::iterator
  CbjSolver::
  label (SearchPath::iterator i, bool &consistent)
  {
    assert (i != get_current_path ().end ());
    assert (*i != NULL);

    // consistent <-- false;
    consistent = false;

    // FOR v[i] <-- EACH ELEMENT OF current-domain[i] WHILE not consistent
    for (Domain::iterator val (get_current_domain_at (*i).begin ()); 
	 !consistent && val != get_current_domain_at (*i).end (); )
      {
	// Instantiate the current variable with the next value in its
	// domain.
	assign (i, *val);

	// Increment the number of nodes visited since a new
	// assignment was made.
	++nv;

	// consistent <-- true;
	consistent = true;

	// FOR h <-- 1 TO i - 1 WHILE consistent
	SearchPath::iterator h;
	SearchPath::const_iterator h_last (get_current_path ().begin ());
	for (h = get_current_path ().begin () + 1; h <= i - 1 && consistent;
	     ++h)
	  {
	    h_last = h;

	    // consistent <-- check (i, h);
	    const VvpSet vvps (*i, get_value (*i), *h, get_value (*h));
	    consistent = problem->check (vvps, NULL);
	    
	    // Increment the total number of constraint checks.
	    ++cc;
	  }
	assert ((h_last == get_current_path ().begin () 
		 && h == get_current_path ().begin () + 1)
		|| h == h_last + 1);

	// IF not consistent:
	if (!consistent)
	  {
	    // pushnew (h - 1, conf-set[i]).
	    insert_into_conf_set_at (*i, h - 1);

	    // current-domain[i] <-- remove (v[i], current-domain[i]).
	    remove_from_current_domain_at (*i, *val++);
	  }
	else
	  {
	    ++val;
	  }
      }
    
    // If consistent THEN return (i + 1) ELSE return (i).
    if (consistent)
      return i + 1;
    else
      return i;
  }

  
  SearchPath::iterator
  CbjSolver::
  unlabel (SearchPath::iterator i, bool &consistent)
  {
    assert (i != get_current_path ().end ());
    assert (*i != NULL);

    // Increment the number of backtracks.
    ++bt;

    // h <-- max-list (conf-set[i]);
    SearchPathIterSet::const_iterator max_conf
      (std::max_element (conf_set_at (*i).begin (),
			 conf_set_at (*i).end ()));
    assert (max_conf != conf_set_at (*i).end ());
    SearchPath::iterator h (*max_conf);

    // conf-set[h] <-- remove (h, union (conf-set[h], conf-set[i]));
    SearchPathIterSet unionx;
    std::insert_iterator<SearchPathIterSet> union_inserter
      (unionx, unionx.end ());
    // Conf-sets are sets, so they are already sorted.
    std::set_union (conf_set_at (*h).begin (),
		    conf_set_at (*h).end (),
		    conf_set_at (*i).begin (),
		    conf_set_at (*i).end (), union_inserter);
    unionx.erase (h);
    set_conf_set_at (*h, unionx);

    // FOR j <-- h + 1 TO i
    for (SearchPath::iterator j (h + 1); j <= i; ++j)
      {
	// conf-set[j] <-- {0};
	reset_conf_set_at (*j);

	// current-domain[j] <-- domain[j]
	reset_current_domain_at (*j);
      }

    // current-domain[h] <-- remove (v[h], current-domain[h]).
    remove_from_current_domain_at (*h, get_value (*h));

    // consistent <-- current-domain[h] /= nil.
    consistent = !get_current_domain_at (*h).empty ();

    // return (h).
    return h;
  }


  SearchPath::iterator
  CbjSolver::
  unlabel_solution (bool &consistent)
  {
    assert (conf_sets.find (*(get_current_path ().end () - 1))
	    != conf_sets.end ());
    assert (!conf_sets.find (*(get_current_path ().end () - 1))
	    ->second.empty ());
    assert (*conf_sets.find (*(get_current_path ().end () - 1))->second.begin ()
	    == get_current_path ().begin ());

    // Increment the number of backtracks.
    ++bt;

    SearchPath::iterator h (get_current_path ().end () - 1);

    // Force the last variable to conflict with everything before it.
    SearchPathIterSet conf_set;
    for (SearchPath::iterator p (get_current_path ().begin ());
	 p != h; ++p)
      conf_set.insert (p);
    set_conf_set_at (*h, conf_set);

    for (SearchPath::iterator j = h + 1; j < get_current_path ().end (); ++j)
      {
	assert (false);
	reset_conf_set_at (*j);
	reset_current_domain_at (*j);
      }
    remove_from_current_domain_at (*h, get_value (*h));
    consistent = !get_current_domain_at (*h).empty ();
    return h;
  }


  std::string 
  CbjSolver::
  algorithm_string (void) const
  {
    return "CBJ";
  }


  // Initialize each chronological backtracking flag to false.
  void
  CbjCbfSolver::
  init_cbfs (void)
  {
    assert (cbfs.empty ());

    for (VariableVector::const_iterator i (problem->get_variables ().begin ()); 
	 i != problem->get_variables ().end (); ++i)
      {
	assert (*i != NULL);
	assert (cbfs.find (*i) == cbfs.end ());

	cbfs.insert (std::make_pair (*i, false));

	assert (cbfs.find (*i) != cbfs.end ());
      }
      
    assert (cbfs.size () == num_vars ());
  }


  bool
  CbjCbfSolver::
  cbf_at (const SearchPath::value_type &var) const
  {
    assert (cbfs.find (var) != cbfs.end ());
    assert (cbfs.size () == num_vars ());

    return cbfs.find (var)->second;
  }


  void
  CbjCbfSolver::
  set_all_cbfs (void)
  {
    assert (cbfs.size () == num_vars ());

    for (BacktrackFlags::iterator i (cbfs.begin ()); i != cbfs.end (); ++i)
      i->second = true;

    assert (cbfs.size () == num_vars ());
  }

  
  void
  CbjCbfSolver::
  zero_cbf_at (const SearchPath::value_type &var)
  {
    assert (cbfs.find (var) != cbfs.end ());
    assert (cbfs.size () == num_vars ());

    cbfs[var] = false;

    assert (cbfs.find (var) != cbfs.end ());
    assert (cbfs.size () == num_vars ());
  }


  // This method is based on Prosser's cbj-label.
  SearchPath::iterator
  CbjCbfSolver::
  label (SearchPath::iterator i, bool &consistent)
  {
    assert (i != get_current_path ().end ());
    assert (*i != NULL);

    // consistent <-- false;
    consistent = false;

    // FOR v[i] <-- EACH ELEMENT OF current-domain[i] WHILE not consistent
    for (Domain::iterator val (get_current_domain_at (*i).begin ()); 
	 !consistent && val != get_current_domain_at (*i).end (); )
      {
	// Instantiate the current variable with the next value in its
	// domain.
	assign (i, *val);

	// Increment the number of nodes visited since a new
	// assignment was made.
	++nv;

	// consistent <-- true;
	consistent = true;

	// FOR h <-- 1 TO i - 1 WHILE consistent
	SearchPath::iterator h;
	SearchPath::const_iterator h_last (get_current_path ().begin ());
	for (h = get_current_path ().begin () + 1; h <= i - 1 && consistent;
	     ++h)
	  {
	    h_last = h;

	    // consistent <-- check (i, h);
	    const VvpSet vvps (*i, get_value (*i), *h, get_value (*h));
	    consistent = problem->check (vvps, NULL);
	    
	    // Increment the total number of constraint checks.
	    ++cc;
	  }
	assert ((h_last == get_current_path ().begin () 
		 && h == get_current_path ().begin () + 1)
		|| h == h_last + 1);

	// IF not consistent:
	if (!consistent)
	  {
	    // pushnew (h - 1, conf-set[i]).
	    insert_into_conf_set_at (*i, h - 1);

	    // current-domain[i] <-- remove (v[i], current-domain[i]).
	    remove_from_current_domain_at (*i, *val++);
	  }
	else
	  {
	    ++val;
	  }
      }
    
    // If consistent THEN return (i + 1) ELSE return (i).
    if (consistent)
      {
	return i + 1;
      }
    else
      return i;
  }

  
  SearchPath::iterator
  CbjCbfSolver::
  unlabel (SearchPath::iterator i, bool &consistent)
  {
    assert (i != get_current_path ().end ());
    assert (*i != NULL);

    // Increment the number of backtracks.
    ++bt;

    // If the CBF is set for *I, we use the previously instantiated
    // variable; otherwise, proceed as normal.
    SearchPath::iterator h;
    if (cbf_at (*i))
      {
	// h <-- i - 1;
	h = i - 1;

	// Zero the CBF for this variable.
	zero_cbf_at (*i);
      }
    else
      {
	// h <-- max-list (conf-set[i]);
	SearchPathIterSet::const_iterator max_conf
	  (std::max_element (conf_set_at (*i).begin (),
			     conf_set_at (*i).end ()));
	assert (max_conf != conf_set_at (*i).end ());
	h = *max_conf;
      }

    // conf-set[h] <-- remove (h, union (conf-set[h], conf-set[i]));
    SearchPathIterSet unionx;
    std::insert_iterator<SearchPathIterSet> union_inserter
      (unionx, unionx.end ());
    // Conf-sets are sets, so they are already sorted.
    std::set_union (conf_set_at (*h).begin (),
		    conf_set_at (*h).end (),
		    conf_set_at (*i).begin (),
		    conf_set_at (*i).end (), union_inserter);
    unionx.erase (h);
    set_conf_set_at (*h, unionx);

    // FOR j <-- h + 1 TO i
    for (SearchPath::iterator j (h + 1); j <= i; ++j)
      {
	// conf-set[j] <-- {0};
	reset_conf_set_at (*j);

	// current-domain[j] <-- domain[j]
	reset_current_domain_at (*j);
      }

    // current-domain[h] <-- remove (v[h], current-domain[h]).
    remove_from_current_domain_at (*h, get_value (*h));

    // consistent <-- current-domain[h] /= nil.
    consistent = !get_current_domain_at (*h).empty ();

    // return (h).
    return h;
  }


  SearchPath::iterator
  CbjCbfSolver::
  unlabel_solution (bool &consistent)
  {
    // Increment the number of backtracks.
    ++bt;

    // Set the CBF for all variables.
    set_all_cbfs ();

    SearchPath::iterator h (get_current_path ().end () - 1);

    for (SearchPath::iterator j = h + 1; j < get_current_path ().end (); ++j)
      {
	reset_conf_set_at (*j);
	reset_current_domain_at (*j);
      }
    remove_from_current_domain_at (*h, get_value (*h));
    consistent = !get_current_domain_at (*h).empty ();
    return h;
  }


  std::string 
  CbjCbfSolver::
  algorithm_string (void) const
  {
    return "CBJ-CBF";
  }

  
  SearchPath::iterator
  FcCbjSolver::
  label (SearchPath::iterator i, bool &consistent)
  {
    assert (i != get_current_path ().end ());
    assert (*i != NULL);

    // Assume that we are inconsistent.
    consistent = false;
    
    for (Domain::iterator val (get_current_domain_at (*i).begin ()); 
	 !consistent && val != get_current_domain_at (*i).end (); )
      {
	// Instantiate the current variable with the next value in its
	// domain.
	assign (i, *val);
	
	// Increment the number of nodes visited since a new
	// assignment was made.
	++nv;

	// The current assignments are consistent unless we say
	// otherwise below.
	consistent = true;

	SearchPath::iterator j;
	SearchPath::const_iterator j_last (get_current_path ().begin ());
	for (j = i + 1; j < get_current_path ().end () && consistent; ++j)
	  {
	    j_last = j;
	    consistent = check_forward (i, j);
	  }
	assert (j_last == get_current_path().begin () && j == i + 1
		|| j == j_last + 1);
	
	if (!consistent)
	  {
	    remove_from_current_domain_at (*i, *val++);
	    undo_reductions (i);
	    
	    // We must sort PAST_FC[J - 1] before we perform the
	    // union, but first it must be copied since we cannot
	    // change the current order of PAST_FC[J - 1].
	    PastFcsDeque sorted_past_fc (get_past_fcs_at (*(j - 1)));
	    std::sort (sorted_past_fc.begin (), sorted_past_fc.end ());

	    // Now do the union.
	    SearchPathIterSet unionx;
	    // Const-sets are sets, so they are already sorted.
	    std::insert_iterator<SearchPathIterSet> union_inserter
	      (unionx, unionx.end ());
	    std::set_union (conf_set_at (*i).begin (),
			    conf_set_at (*i).end (),
			    sorted_past_fc.begin (),
			    sorted_past_fc.end (), union_inserter);
	    set_conf_set_at (*i, unionx);
	  }
	else
	  {
	    ++val;
	  }
      }

    // If we are consistent, then return an index to the next future
    // variable; otherwise, return an index to the current variable.
    if (consistent)
      return i + 1;
    else
      return i;
  }

  SearchPath::iterator
  FcCbjSolver::
  unlabel (SearchPath::iterator i, bool &consistent)
  {
    assert (i != get_current_path ().end ());
    assert (*i != NULL);

    // Increment the number of backtracks.
    ++bt;

    // h <-- max (max-list (conf-set[i]), max-list (past-fc[i])) (with
    // special handling in case one of the lists is empty).
    SearchPath::iterator h;
    SearchPathIterSet::const_iterator max_conf 
      (std::max_element (conf_set_at (*i).begin (), 
			 conf_set_at (*i).end ()));
    PastFcsDeque::const_iterator max_past
      (std::max_element (get_past_fcs_at (*i).begin (),
			 get_past_fcs_at (*i).end ()));
    if (max_conf != conf_set_at (*i).end ()
	&& max_past != get_past_fcs_at (*i).end ())
      h = std::max (*max_conf, *max_past);
    else if (max_conf != conf_set_at (*i).end () 
	     && max_past == get_past_fcs_at (*i).end ())
      h = *max_conf;
    else if (max_conf == conf_set_at (*i).end () 
	     && max_past != get_past_fcs_at (*i).end ())
      h = *max_past;
    else
      assert (false);

    // conf-set[h] <-- remove (h, union (conf-set[h],
    //                                   union (conf-set[i], past-fc[i])))
    // First, find the inner union (union (conf-set[i], past-fc[i])).
    // This requires sorting a copy of PAST_FC[i] first.
    PastFcsDeque sorted_past_fc (get_past_fcs_at (*i));
    std::sort (sorted_past_fc.begin (), sorted_past_fc.end ());
    SearchPathIterSet conf_past_union;
    std::insert_iterator<SearchPathIterSet> conf_past_inserter
      (conf_past_union, conf_past_union.end ());
    std::set_union (conf_set_at (*i).begin (),
		    conf_set_at (*i).end (),
		    sorted_past_fc.begin (),
		    sorted_past_fc.end (), conf_past_inserter);
    // Now find the outer union.
    SearchPathIterSet conf_conf_past_union;
    std::insert_iterator<SearchPathIterSet> conf_conf_past_inserter
      (conf_conf_past_union, conf_conf_past_union.end ());
    std::set_union (conf_set_at (*h).begin (),
		    conf_set_at (*h).end (),
		    conf_past_union.begin (),
		    conf_past_union.end (), conf_conf_past_inserter);
    // Finally, remove h and make the assignment.
    if (h != get_current_path ().begin ()) // TODO: Is this needed?
      conf_conf_past_union.erase (h);
    set_conf_set_at (*h, conf_conf_past_union);
    
    // FOR j <-- i DOWNTO h + 1
    for (SearchPath::iterator j (i); j >= h + 1; --j)
      {
	// conf-set[j] <-- {0}
	reset_conf_set_at (*j);
	
	undo_reductions (j);
	update_current_domain (j);
      }

    undo_reductions (h);

    // current-domain[h] <-- remove (v[h], current-domain[h])
    remove_from_current_domain_at (*h, get_value (*h));

    // consistent <-- current-domain[h] /= nil
    consistent = !get_current_domain_at (*h).empty ();

    return h;
  }
  

  SearchPath::iterator
  FcCbjSolver::
  unlabel_solution (bool &consistent)
  {
    // Increment the number of backtracks.
    ++bt;

    SearchPath::iterator h (get_current_path ().end () - 1);

    // Force the last variable to conflict with everything before it.
    SearchPathIterSet conf_set;
    for (SearchPath::iterator p (get_current_path ().begin ());
	 p != h; ++p)
      conf_set.insert (p);
    set_conf_set_at (*h, conf_set);
    
    undo_reductions (h);
    remove_from_current_domain_at (*h, get_value (*h));
    consistent = !get_current_domain_at (*h).empty ();
    return h;
  }


  std::string 
  FcCbjSolver::
  algorithm_string (void) const
  {
    return "FC-CBJ";
  }


  // The functions below are only used for debugging.


  std::ostream &
  operator<< (std::ostream &out, const SearchPath &path)
  {
    out << '[';
    for (SearchPath::const_iterator i (path.begin ()); i != path.end (); ++i)
      {
	if (i != path.begin ())
	  out << ", ";
	assert (*i != NULL);
	out << **i;
      }
    out << ']';
    return out;
  }


  // An output operator for assignments.
  std::ostream &
  operator<< (std::ostream &out, const Assignments &assignments)
  {
    out << '{';
    for (Assignments::const_iterator i (assignments.begin ());
	 i != assignments.end (); ++i)
      {
	if (i != assignments.begin ())
	  out << ", ";
	assert (i->first != NULL);
	out << '(' << *i->first << ',' << i->second << ')';
      }
    out << '}';
    return out;
  }


  // An output operator for current domains.
  std::ostream &
  operator<< (std::ostream &out, const CurrentDomains &domains)
  {
    out << '{';
    for (CurrentDomains::const_iterator i (domains.begin ());
	 i != domains.end (); ++i)
      {
	if (i != domains.begin ())
	  out << ", ";
	assert (i->first != NULL);
	out << '(' << *i->first << ',' << i->second << ')';
      }
    out << '}';
    return out;
  }
  

  // An output operator for SearchPathIterSets.
  std::ostream &
  operator<< (std::ostream &out, const SearchPathIterSet &set)
  {
    out << '{';
    for (SearchPathIterSet::const_iterator i (set.begin ());
	 i != set.end (); ++i)
      {
	if (i != set.begin ())
	  out << ", ";
	assert (**i != NULL);
	out << ***i;
      }
    out << '}';
    return out;
  }
} // namespace CSPSolver
